<?php
/**
 * REST API: EE_REST_Garbage_Centers_Controller class
 *
 * @package Easyeco New
 * @since 1.0.0
 */

/**
 * Class to access garbage centers via the REST API.
 *
 * @since 1.0.0
 *
 * @see WP_REST_Posts_Controller
 */
class EE_REST_Garbage_Centers_Controller extends WP_REST_Posts_Controller
{
    /**
     * Post type.
     *
     * @since 1.0.0
     * @var string
     */
    protected $post_type;

    /**
     * Instance of a post meta fields object.
     *
     * @since 1.0.0
     * @var WP_REST_Post_Meta_Fields
     */
    protected $meta;

    /**
     * Constructor.
     *
     * @since 1.0.0
     *
     * @param string $post_type Post type.
     */
    public function __construct()
    {
        parent::__construct('garbage_center');
        $this->namespace = 'ee/v1';
    }

    /**
     * Registers the routes for the objects of the controller.
     *
     * @since 1.0.0
     *
     * @see register_rest_route()
     */
    public function register_routes()
    {

        register_rest_route(
            $this->namespace,
            '/' . $this->rest_base,
            array(
                array(
                    'methods' => WP_REST_Server::READABLE,
                    'callback' => array($this, 'get_items'),
                    'permission_callback' => array($this, 'get_items_permissions_check'),
                ),
                'schema' => array($this, 'get_public_item_schema'),
            )
        );

        register_rest_route(
            $this->namespace,
            '/' . $this->rest_base . '/(?P<id>[\d]+)',
            array(
                'args' => array(
                    'id' => array(
                        'description' => __('Unique identifier for the object.'),
                        'type' => 'integer',
                    ),
                ),
                array(
                    'methods' => WP_REST_Server::READABLE,
                    'callback' => array($this, 'get_item'),
                    'permission_callback' => array($this, 'get_item_permissions_check'),
                ),
                'schema' => array($this, 'get_public_item_schema'),
            )
        );
    }

    /**
     * Retrieves a collection of posts.
     *
     * @since 1.0.0
     *
     * @param WP_REST_Request $request Full details about the request.
     * @return WP_REST_Response|WP_Error Response object on success, or WP_Error object on failure.
     */
    public function get_items($request)
    {
        $zone_id = $request->get_header('zone-id');
        // Ensure a search string is set in case the orderby is set to 'relevance'.
        if (!$zone_id) {
            return new WP_Error(
                'rest_no_zone_id_header_defined',
                __('You need to define a zone id header.'),
                array('status' => 400)
            );
        }

        $args = array();

        $args['post_type'] = $this->post_type;
        $args['posts_per_page'] = 100;

        $query_args = $this->prepare_items_query($args, $request);
        $query_args['tax_query'][] = array(
            'taxonomy' => 'municipality',
            'field' => 'term_id',
            'terms' => $zone_id,
            'include_children' => false,
        );

        $posts_query = new WP_Query();
        $query_result = $posts_query->query($query_args);

        $posts = array();

        foreach ($query_result as $post) {
            if (!$this->check_read_permission($post)) {
                continue;
            }

            $request['context'] = 'list';
            $data = $this->prepare_item_for_response($post, $request);
            $posts[] = $this->prepare_response_for_collection($data);
        }

        $response = rest_ensure_response($posts);

        $request_params = $request->get_query_params();
        $base = add_query_arg(urlencode_deep($request_params), rest_url(sprintf('%s/%s', $this->namespace, $this->rest_base)));

        return $response;
    }

    /**
     * Retrieves a single post.
     *
     * @since 1.0.0
     *
     * @param WP_REST_Request $request Full details about the request.
     * @return WP_REST_Response|WP_Error Response object on success, or WP_Error object on failure.
     */
    public function get_item($request)
    {
        $post = $this->get_post($request['id']);
        if (is_wp_error($post)) {
            return $post;
        }
        $request['context'] = 'single';
        $data = $this->prepare_item_for_response($post, $request);
        $response = rest_ensure_response($data);

        if (is_post_type_viewable(get_post_type_object($post->post_type))) {
            $response->link_header('alternate', get_permalink($post->ID), array('type' => 'text/html'));
        }

        return $response;
    }

    /**
     * Prepares a single post output for response.
     *
     * @since 1.0.0
     *
     * @param WP_Post         $post    Post object.
     * @param WP_REST_Request $request Request object.
     * @return WP_REST_Response Response object.
     */
    public function prepare_item_for_response($post, $request)
    {
        $GLOBALS['post'] = $post;

        setup_postdata($post);

        // Base fields for every post.
        $data = array();

        $data['id'] = $post->ID;

        $data['slug'] = $post->post_name;

        $data['title'] = array();

        add_filter('protected_title_format', array($this, 'protected_title_format'));
        $data['title']['rendered'] = $post->post_title;
        remove_filter('protected_title_format', array($this, 'protected_title_format'));

        $data['content'] = array();
        $data['content']['rendered'] = post_password_required($post) ? '' : apply_filters('the_content', $post->post_content);

        //CUSTOM
        if (get_post_meta($post->ID, 'view_address', true)) {
            $data['address'] = get_post_meta($post->ID, 'address', true);
            $data['coordinates']['lat'] = get_post_meta($post->ID, 'lat', true);
            $data['coordinates']['lng'] = get_post_meta($post->ID, 'lng', true);
        } else {
            $data['address'] = false;
            $data['coordinates'] = false;
        }

        $context = !empty($request['context']) ? $request['context'] : 'view';

        $data = $this->filter_response_by_context($data, $context);

        // Wrap the data in a response object.
        $response = rest_ensure_response($data);

        /**
         * Filters the post data for a response.
         *
         * The dynamic portion of the hook name, `$this->post_type`, refers to the post type slug.
         *
         * @since 1.0.0
         *
         * @param WP_REST_Response $response The response object.
         * @param WP_Post          $post     Post object.
         * @param WP_REST_Request  $request  Request object.
         */
        return apply_filters("rest_prepare_{$this->post_type}", $response, $post, $request);
    }

    /**
     * Retrieves the post's schema, conforming to JSON Schema.
     *
     * @since 1.0.0
     *
     * @return array Item schema data.
     */
    public function get_item_schema()
    {
        $schema = parent::get_item_schema();
        
        $schema['properties']['id']['context'][] = 'list';
        $schema['properties']['id']['context'][] = 'single';

        $schema['properties']['slug']['context'][] = 'list';
        $schema['properties']['slug']['context'][] = 'single';

        $schema['properties']['title']['context'][] = 'list';
        $schema['properties']['title']['context'][] = 'single';

        $schema['properties']['title']['properties']['rendered']['context'][] = 'list';
        $schema['properties']['title']['properties']['rendered']['context'][] = 'single';

        $schema['properties']['content']['context'][] = 'single';

        $schema['properties']['content']['properties']['rendered']['context'][] = 'single';

        $schema['properties']['address'] = array(
            'description' => __('Address of the garbage center'),
            'type' => 'string',
            'context' => array('single'),
        );
        $schema['properties']['coordinates'] = array(
            'description' => __('Coordinates of the garbage center'),
            'type' => 'string',
            'context' => array('single'),
        );
        $schema['properties']['lat'] = array(
            'description' => __('Latitude of the garbage center'),
            'type' => 'string',
            'context' => array('single'),
        );
        $schema['properties']['lng'] = array(
            'description' => __('Longitude of the garbage center'),
            'type' => 'string',
            'context' => array('single'),
        );
        return $schema;
    }

}
