<?php
/**
 * REST API: EE_REST_Garbages_Controller class
 *
 * @package Easyeco New
 * @since 1.0.0
 */

/**
 * Class to access garbages via the REST API.
 *
 * @since 1.0.0
 *
 * @see WP_REST_Posts_Controller
 */
class EE_REST_Garbages_Controller extends WP_REST_Posts_Controller
{
    /**
     * Post type.
     *
     * @since 1.0.0
     * @var string
     */
    protected $post_type;

    /**
     * Instance of a post meta fields object.
     *
     * @since 1.0.0
     * @var WP_REST_Post_Meta_Fields
     */
    protected $meta;

    /**
     * Constructor.
     *
     * @since 1.0.0
     *
     * @param string $post_type Post type.
     */
    public function __construct()
    {
        parent::__construct('garbage');
		$this->namespace = 'ee/v1';
    }

    /**
     * Registers the routes for the objects of the controller.
     *
     * @since 1.0.0
     *
     * @see register_rest_route()
     */
    public function register_routes()
    {
        register_rest_route(
            $this->namespace,
            '/' . $this->rest_base,
            array(
                array(
                    'methods' => WP_REST_Server::READABLE,
                    'callback' => array($this, 'get_items'),
                    'permission_callback' => array($this, 'get_items_permissions_check'),
                    'args'                => $this->get_collection_params(),
                ),
                'schema' => array($this, 'get_public_item_schema'),
            )
        );

        register_rest_route(
            $this->namespace,
            '/' . $this->rest_base . '/selected',
            array(
                array(
                    'methods' => WP_REST_Server::READABLE,
                    'callback' => array($this, 'get_selected_items'),
                    'permission_callback' => array($this, 'get_items_permissions_check'),
                    //'args'                => $this->get_collection_params(),
                ),
                'schema' => array($this, 'get_public_item_schema'),
            )
        );
    }

    /**
     * Retrieves the query params for the collections.
     *
     * @since 1.0.0
     *
     * @return array Query parameters for the collection.
     */
    public function get_collection_params()
    {
        $query_params = array(
            'search' => array(
                'description' => __('Limit results to those matching a string.'),
                'type' => 'string',
                'sanitize_callback' => 'sanitize_text_field',
                'validate_callback' => 'rest_validate_request_arg',
            ),
            'order' => array(
                'description' => __( 'Order sort attribute ascending or descending.' ),
                'type'        => 'string',
                'default'     => 'desc',
                'enum'        => array( 'asc', 'desc' ),
            ),
            'orderby' => array(
                'description' => __( 'Sort collection by object attribute.' ),
                'type'        => 'string',
                'default'     => 'date',
                'enum'        => array(
                    'author',
                    'date',
                    'id',
                    'include',
                    'modified',
                    'parent',
                    'relevance',
                    'slug',
                    'include_slugs',
                    'title',
                ),
            ),
            'context' => $this->get_context_param(),
        );

        return $query_params;
    }

    /**
     * Retrieves a collection of posts.
     *
     * @since 1.0.0
     *
     * @param WP_REST_Request $request Full details about the request.
     * @return WP_REST_Response|WP_Error Response object on success, or WP_Error object on failure.
     */
    public function get_items($request)
    {
        $request['context'] = 'list';
        $request['orderby'] = 'title';
        $request['order'] = 'asc';
        return parent::get_items($request);
    }

    /**
     * Retrieves a collection of posts.
     *
     * @since 1.0.0
     *
     * @param WP_REST_Request $request Full details about the request.
     * @return WP_REST_Response|WP_Error Response object on success, or WP_Error object on failure.
     */
    public function get_selected_items($request)
    {
        $selected_garbages = get_option('options_selected_garbages');

        $args = array();

        $args['post_type'] = $this->post_type;
        $args['post__in'] = $selected_garbages;
        $args['orderby'] = 'title';
        $args['order'] = 'asc';
        $args['posts_per_page'] = 100;

        $query_args = $this->prepare_items_query($args, $request);

        $posts_query = new WP_Query();
        $query_result = $posts_query->query($query_args);

        $posts = array();

        $request['context'] = 'selected';

        foreach ($query_result as $post) {
            if (!$this->check_read_permission($post)) {
                continue;
            }

            $data = $this->prepare_item_for_response($post, $request);
            $posts[] = $this->prepare_response_for_collection($data);
        }

        $response = rest_ensure_response($posts);

        $request_params = $request->get_query_params();
        $base = add_query_arg(urlencode_deep($request_params), rest_url(sprintf('%s/%s', $this->namespace, $this->rest_base)));

        return $response;
    }

    /**
     * Prepares a single post output for response.
     *
     * @since 1.0.0
     *
     * @param WP_Post         $post    Post object.
     * @param WP_REST_Request $request Request object.
     * @return WP_REST_Response Response object.
     */
    public function prepare_item_for_response($post, $request)
    {
        $GLOBALS['post'] = $post;

        setup_postdata($post);

        // Base fields for every post.
        $data = array();

        $data['id'] = $post->ID;

        $data['slug'] = $post->post_name;

        $data['title'] = array();
        add_filter('protected_title_format', array($this, 'protected_title_format'));

        $data['title']['rendered'] = get_the_title($post->ID);
        remove_filter('protected_title_format', array($this, 'protected_title_format'));
        $context = !empty($request['context']) ? $request['context'] : 'view';

        $data = $this->filter_response_by_context($data, $context);

        // Wrap the data in a response object.
        $response = rest_ensure_response($data);

        /**
         * Filters the post data for a response.
         *
         * The dynamic portion of the hook name, `$this->post_type`, refers to the post type slug.
         *
         * @since 1.0.0
         *
         * @param WP_REST_Response $response The response object.
         * @param WP_Post          $post     Post object.
         * @param WP_REST_Request  $request  Request object.
         */
        return apply_filters("rest_prepare_{$this->post_type}", $response, $post, $request);
    }

    /**
     * Retrieves the post's schema, conforming to JSON Schema.
     *
     * @since 1.0.0
     *
     * @return array Item schema data.
     */
    public function get_item_schema()
    {
        $schema = parent::get_item_schema();
        
        $schema['properties']['id']['context'][] = 'list';
        $schema['properties']['id']['context'][] = 'selected';

        $schema['properties']['slug']['context'][] = 'list';
        $schema['properties']['slug']['context'][] = 'selected';

        $schema['properties']['title']['context'][] = 'list';
        $schema['properties']['title']['context'][] = 'selected';

        $schema['properties']['title']['properties']['rendered']['context'][] = 'list';
        $schema['properties']['title']['properties']['rendered']['context'][] = 'selected';
        return $schema;
    }

}
