<?php
/**
 * REST API: EE_REST_Municipalities_Controller class
 *
 * @package Easyeco New
 * @since 1.0.0
 */

/**
 * Class to access municipalities via the REST API.
 *
 * @since 1.0.0
 *
 * @see WP_REST_Taxonomies_Controller
 */
class EE_REST_Municipalities_Controller extends WP_REST_Taxonomies_Controller
{

    /**
     * Constructor.
     *
     * @since 1.0.0
     */
    public function __construct()
    {
        $this->namespace = 'ee/v1';
        $this->rest_base = 'municipalities';
    }

    /**
     * Registers the routes for the objects of the controller.
     *
     * @since 1.0.0
     *
     * @see register_rest_route()
     */
    public function register_routes()
    {

        register_rest_route(
            $this->namespace,
            '/' . $this->rest_base,
            array(
                array(
                    'methods' => WP_REST_Server::READABLE,
                    'callback' => array($this, 'get_items'),
                    'permission_callback' => array($this, 'get_items_permissions_check'),
                    'args' => $this->get_collection_params(),
                ),
                'schema' => array($this, 'get_public_item_schema'),
            )
        );

        register_rest_route(
            $this->namespace,
            '/' . $this->rest_base . '/(?P<id>[\d]+)/streets',
            array(
                'args' => array(
                    'id' => array(
                        'description' => __('Unique identifier for the object.'),
                        'type' => 'integer',
                    ),
                ),
                array(
                    'methods' => WP_REST_Server::READABLE,
                    'callback' => array($this, 'get_streets'),
                    'permission_callback' => array($this, 'get_item_permissions_check'),
                ),
                'schema' => array($this, 'get_public_item_schema'),
            )
        );
    }

    /**
     * Retrieves the query params for the collections.
     *
     * @since 1.0.0
     *
     * @return array Query parameters for the collection.
     */
    public function get_collection_params()
    {
        $query_params = array(
            'search' => array(
                'description' => __('Limit results to those matching a string.'),
                'type' => 'string',
                'sanitize_callback' => 'sanitize_text_field',
                'validate_callback' => 'rest_validate_request_arg',
            ),
            'context' => $this->get_context_param(),
        );

        return $query_params;
    }

    /**
     * Retrieves all public taxonomies.
     *
     * @since 1.0.0
     *
     * @param WP_REST_Request $request Full details about the request.
     * @return WP_REST_Response Response object on success, or WP_Error object on failure.
     */
    public function get_items($request)
    {
        $terms = get_terms(array(
            'orderby' => 'name',
            'order' => 'ASC',
            'taxonomy' => 'municipality',
            'hide_empty' => false,
            'search' => isset($request['search']) ? $request['search'] : '',
            'parent' => 0,
            'number' => 1000,
            'hierarchical' => false,
        ));

        $data = array();

        $request['context'] = 'list';

        foreach ($terms as $key => $value) {
            $term = $this->prepare_item_for_response($value, $request);
            $term = $this->prepare_response_for_collection($term);
            $data[$key] = $term;
        }

        if (empty($data)) {
            // Response should still be returned as a JSON object when it is empty.
            $data = (object) $data;
        }

        return rest_ensure_response($data);
    }

    /**
     * Retrieves a specific taxonomy.
     *
     * @since 1.0.0
     *
     * @param WP_REST_Request $request Full details about the request.
     * @return WP_REST_Response|WP_Error Response object on success, or WP_Error object on failure.
     */
    public function get_streets($request)
    {
        $query_args['post_type'] = 'street';
        $query_args['s'] = isset($request['search']) ? $request['search'] : '';
        $query_args['posts_per_page'] = (isset($request['per_page']) ? $request['per_page'] : 50);
        $query_args['orderby'] = 'title';
        $query_args['paged'] = $request['page'];

        $query_args['tax_query'][] = array(
            'taxonomy' => 'municipality',
            'field' => 'term_id',
            'terms' => $request['id'],
            'include_children' => true,
        );

        $posts_query = new WP_Query();
        $query_result = $posts_query->query($query_args);

        $posts = array();

        $request['context'] = 'street';

        foreach ($query_result as $post) {
            $data = $this->prepare_streets_for_response($post, $request);
            $posts[] = $this->prepare_response_for_collection($data);
        }

        $total_posts = $posts_query->found_posts;
 
        if ( $total_posts < 1 ) {
            // Out-of-bounds, run the query again without LIMIT for total count.
            unset( $query_args['paged'] );
 
            $count_query = new WP_Query();
            $count_query->query( $query_args );
            $total_posts = $count_query->found_posts;
        }
 
        $max_pages = ceil( $total_posts / (int) $posts_query->query_vars['posts_per_page'] );
        $page = (int) $query_args['paged'];
        if ( $page > $max_pages && $total_posts > 0 ) {
            return new WP_Error(
                'rest_post_invalid_page_number',
                __( 'The page number requested is larger than the number of pages available.' ),
                array( 'status' => 400 )
            );
        }
 
        $response = rest_ensure_response( $posts );
 
        $response->header( 'X-WP-Total', (int) $total_posts );
        $response->header( 'X-WP-TotalPages', (int) $max_pages );
 
        $request_params = $request->get_query_params();
        $base = add_query_arg( urlencode_deep( $request_params ), rest_url( sprintf( '%s/%s', $this->namespace, $this->rest_base ) ) );
 
        if ( $page > 1 ) {
            $prev_page = $page - 1;
 
            if ( $prev_page > $max_pages ) {
                $prev_page = $max_pages;
            }
 
            $prev_link = add_query_arg( 'page', $prev_page, $base );
            $response->link_header( 'prev', $prev_link );
        }
        if ( $max_pages > $page ) {
            $next_page = $page + 1;
            $next_link = add_query_arg( 'page', $next_page, $base );
 
            $response->link_header( 'next', $next_link );
        }
 
        return $response;
    }

    /**
     * Prepares a term object for serialization.
     *
     * @since 1.0.0
     *
     * @param WP_Term     $term Term data.
     * @param WP_REST_Request $request  Full details about the request.
     * @return WP_REST_Response Response object.
     */
    public function prepare_item_for_response($term, $request)
    {
        $data = array();

        $data['id'] = $term->term_id;

        $data['slug'] = $term->slug;

        $data['name'] = $term->name;

        $data['zip'] = get_term_meta($term->term_id, 'zip', true);
        
        $context = !empty($request['context']) ? $request['context'] : 'view';

        $data = $this->filter_response_by_context($data, $context);

        // Wrap the data in a response object.
        return rest_ensure_response($data);
    }



    /**
     * Prepares a single post output for response.
     *
     * @since 1.0.0
     *
     * @param WP_Post         $post    Post object.
     * @param WP_REST_Request $request Request object.
     * @return WP_REST_Response Response object.
     */
    public function prepare_streets_for_response($post, $request)
    {
        $GLOBALS['post'] = $post;

        setup_postdata($post);

        // Base fields for every post.
        $data = array();

        $data['id'] = $post->ID;

        $data['slug'] = $post->post_name;

        $data['title'] = array();
        add_filter('protected_title_format', array($this, 'protected_title_format'));

        $data['title']['rendered'] = get_the_title($post->ID);
        remove_filter('protected_title_format', array($this, 'protected_title_format'));

        if($zone = get_the_terms($post->ID, 'municipality')){
            $data['zone-id'] = $zone[0]->term_id;
        } else {
            $data['zone-id'] = false;
        }

        $context = !empty($request['context']) ? $request['context'] : 'view';

        $data = $this->filter_response_by_context($data, $context);

        // Wrap the data in a response object.
        $response = rest_ensure_response($data);

        /**
         * Filters the post data for a response.
         *
         * The dynamic portion of the hook name, `$this->post_type`, refers to the post type slug.
         *
         * @since 1.0.0
         *
         * @param WP_REST_Response $response The response object.
         * @param WP_Post          $post     Post object.
         * @param WP_REST_Request  $request  Request object.
         */
        return apply_filters("rest_prepare_{$this->post_type}", $response, $post, $request);
    }

    /**
     * Retrieves the post's schema, conforming to JSON Schema.
     *
     * @since 1.0.0
     *
     * @return array Item schema data.
     */
    public function get_item_schema()
    {
        $schema = parent::get_item_schema();

        $schema['properties']['id']['context'][] = 'list';
        $schema['properties']['id']['context'][] = 'street';

        $schema['properties']['slug']['context'][] =  'list';
        $schema['properties']['slug']['context'][] = 'street';

        $schema['properties']['name']['context'][] = 'list';

        $schema['properties']['zip'] = array(
            'description' => __('Zip code of the municipality'),
            'type' => 'string',
            'context' => array('list'),
        );
        $schema['properties']['zone-id'] = array(
            'description' => __('Municipality id of the street'),
            'type' => 'string',
            'context' => array('street'),
        );
        
        $schema['properties']['title']['context'][] = 'street';

        $schema['properties']['title']['properties']['rendered']['context'][] = 'street';

        return $schema;
    }

}
