<?php
/**
 * REST API: EE_REST_Pages_Controller class
 *
 * @package Easyeco New
 * @since 1.0.0
 */

/**
 * Core class to access posts via the REST API.
 *
 * @since 1.0.0
 *
 * @see WP_REST_Posts_Controller
 */
class EE_REST_Pages_Controller extends WP_REST_Posts_Controller
{
    /**
     * Post type.
     *
     * @since 1.0.0
     * @var string
     */
    protected $post_type;

    /**
     * Instance of a post meta fields object.
     *
     * @since 1.0.0
     * @var WP_REST_Post_Meta_Fields
     */
    protected $meta;

    /**
     * Constructor.
     *
     * @since 1.0.0
     *
     * @param string $post_type Post type.
     */
    public function __construct()
    {
        parent::__construct('page');
		$this->namespace = 'ee/v1';
    }

    /**
     * Registers the routes for the objects of the controller.
     *
     * @since 1.0.0
     *
     * @see register_rest_route()
     */
    public function register_routes()
    {

        register_rest_route(
            $this->namespace,
            '/' . $this->rest_base,
            array(
                array(
                    'methods' => WP_REST_Server::READABLE,
                    'callback' => array($this, 'get_items'),
                    'permission_callback' => array($this, 'get_items_permissions_check'),
                ),
                'schema' => array($this, 'get_public_item_schema'),
            )
        );

        register_rest_route(
            $this->namespace,
            '/' . $this->rest_base . '/featured',
            array(
                array(
                    'methods' => WP_REST_Server::READABLE,
                    'callback' => array($this, 'get_featured_items'),
                    'permission_callback' => array($this, 'get_items_permissions_check'),
                ),
                'schema' => array($this, 'get_public_item_schema'),
            )
        );

        register_rest_route(
            $this->namespace,
            '/' . $this->rest_base . '/(?P<id>[\d]+)',
            array(
                'args' => array(
                    'id' => array(
                        'description' => __('Unique identifier for the object.'),
                        'type' => 'integer',
                    ),
                ),
                array(
                    'methods' => WP_REST_Server::READABLE,
                    'callback' => array($this, 'get_item'),
                    'permission_callback' => array($this, 'get_item_permissions_check'),
                ),
                'schema' => array($this, 'get_public_item_schema'),
            )
        );
    }

    /**
     * Retrieves a collection of posts.
     *
     * @since 1.0.0
     *
     * @param WP_REST_Request $request Full details about the request.
     * @return WP_REST_Response|WP_Error Response object on success, or WP_Error object on failure.
     */
    public function get_items($request)
    {
        $args = array();

        $zone_id = $request->get_header('zone-id');
        
        $args['post_type'] = $this->post_type;
        $args['posts_per_page'] = 100;
        $args['meta_query'] = $this->get_pages_metaquery($zone_id);

        $query_args = $this->prepare_items_query($args, $request);

        $posts_query = new WP_Query();
        $query_result = $posts_query->query($query_args);

        $posts = array();

        $request['context'] = 'list';
    
        foreach ($query_result as $post) {
            if (!$this->check_read_permission($post)) {
                continue;
            }

            $data = $this->prepare_item_for_response($post, $request);
            $posts[] = $this->prepare_response_for_collection($data);
        }

        $response = rest_ensure_response($posts);

        $request_params = $request->get_query_params();
        $base = add_query_arg(urlencode_deep($request_params), rest_url(sprintf('%s/%s', $this->namespace, $this->rest_base)));

        return $response;
    }

    public function get_pages_metaquery($zone_id){
        if(!empty($zone_id)){
            $withdrawal_service = get_term_meta($zone_id, 'withdrawal_service', true);
            //default withdrawal service is true
            $withdrawal_service = $withdrawal_service == "" ? true : (bool)$withdrawal_service;
            if(!$withdrawal_service){
                return array(
                    array(
                        'key'     => 'template',
                        'value'   => 'withdrawal',
                        'compare' => 'NOT LIKE',
                    ),
                );
            }
        }
        return array();
    }

    /**
     * Retrieves a collection of posts.
     *
     * @since 1.0.0
     *
     * @param WP_REST_Request $request Full details about the request.
     * @return WP_REST_Response|WP_Error Response object on success, or WP_Error object on failure.
     */
    public function get_featured_items($request)
    {
        $posts = array();

        $zone_id = $request->get_header('zone-id');
        
        $featured_pages = get_option('options_featured_pages');

        if($featured_pages){
            $args = array();

            $args['post_type'] = $this->post_type;
            $args['post__in'] = $featured_pages;
            $args['posts_per_page'] = 100;
            $args['meta_query'] = $this->get_pages_metaquery($zone_id);
    
            $query_args = $this->prepare_items_query($args, $request);
    
            $posts_query = new WP_Query();
            $query_result = $posts_query->query($query_args);

            $request['context'] = 'list';
    
            foreach ($query_result as $post) {
                if (!$this->check_read_permission($post)) {
                    continue;
                }
    
                $data = $this->prepare_item_for_response($post, $request);
                $posts[] = $this->prepare_response_for_collection($data);
            }
        }
        

        $response = rest_ensure_response($posts);

        $request_params = $request->get_query_params();
        $base = add_query_arg(urlencode_deep($request_params), rest_url(sprintf('%s/%s', $this->namespace, $this->rest_base)));

        return $response;
    }

    /**
     * Retrieves a single post.
     *
     * @since 1.0.0
     *
     * @param WP_REST_Request $request Full details about the request.
     * @return WP_REST_Response|WP_Error Response object on success, or WP_Error object on failure.
     */
    public function get_item($request)
    {
        $post = $this->get_post($request['id']);
        if (is_wp_error($post)) {
            return $post;
        }

        $request['context'] = 'single';
    
        $data = $this->prepare_item_for_response($post, $request);
        $response = rest_ensure_response($data);

        if (is_post_type_viewable(get_post_type_object($post->post_type))) {
            $response->link_header('alternate', get_permalink($post->ID), array('type' => 'text/html'));
        }

        return $response;
    }

    /**
     * Prepares a single post output for response.
     *
     * @since 1.0.0
     *
     * @param WP_Post         $post    Post object.
     * @param WP_REST_Request $request Request object.
     * @return WP_REST_Response Response object.
     */
    public function prepare_item_for_response($post, $request)
    {
        $GLOBALS['post'] = $post;

        setup_postdata($post);

        // Base fields for every post.
        $data = array();

        $data['id'] = $post->ID;

        $data['slug'] = $post->post_name;

        $data['title'] = array();

        add_filter('protected_title_format', array($this, 'protected_title_format'));
        $data['title']['rendered'] = get_the_title($post->ID);
        remove_filter('protected_title_format', array($this, 'protected_title_format'));

        $data['link'] = get_permalink( $post->ID );

        $data['content'] = array();
        $data['content']['rendered'] = post_password_required($post) ? '' : apply_filters('the_content', $post->post_content);

        $image_preview_id = get_post_meta($post->ID, 'image_preview', true);
        if($image_preview_id){
            $data['image_preview'] = wp_get_attachment_url( $image_preview_id );
        } else {
            $data['image_preview'] = false;
        }
        
        $template = get_post_meta($post->ID, 'template', true);
        if ( $template) {
            if(is_array($template)){
                $data['template'] = $template[0];
            } else{
                $data['template'] = $template;
            }
        } else {
            $data['template'] = '';
        }

        $data['meta'] = $this->meta->get_value( $post->ID, $request );

        $icon_id = get_post_meta($post->ID, 'icon', true);
        if($icon_id){
            $data['icon'] = wp_get_attachment_url( $icon_id );
        } else {
            $data['icon'] = false;
        }

        $context = !empty($request['context']) ? $request['context'] : 'view';

        $data = $this->filter_response_by_context($data, $context);

        // Wrap the data in a response object.
        $response = rest_ensure_response($data);

        /**
         * Filters the post data for a response.
         *
         * The dynamic portion of the hook name, `$this->post_type`, refers to the post type slug.
         *
         * @since 1.0.0
         *
         * @param WP_REST_Response $response The response object.
         * @param WP_Post          $post     Post object.
         * @param WP_REST_Request  $request  Request object.
         */
        return apply_filters("rest_prepare_{$this->post_type}", $response, $post, $request);
    }

    /**
     * Retrieves the post's schema, conforming to JSON Schema.
     *
     * @since 1.0.0
     *
     * @return array Item schema data.
     */
    public function get_item_schema()
    {
        $schema = parent::get_item_schema();
        
        $schema['properties']['id']['context'][] = 'list';
        $schema['properties']['id']['context'][] = 'single';

        $schema['properties']['slug']['context'][] = 'list';
        $schema['properties']['slug']['context'][] = 'single';

        $schema['properties']['link']['context'][] = 'single';

        $schema['properties']['template']['context'][] = 'list';
        $schema['properties']['template']['context'][] = 'single';

        $schema['properties']['meta']['context'][] = 'single';

        $schema['properties']['title']['context'][] = 'list';
        $schema['properties']['title']['context'][] = 'single';

        $schema['properties']['title']['properties']['rendered']['context'][] = 'list';
        $schema['properties']['title']['properties']['rendered']['context'][] = 'single';

        $schema['properties']['content']['context'][] = 'single';
        $schema['properties']['content']['properties']['rendered']['context'][] = 'single';

        $schema['properties']['image_preview'] = array(
            'description' => __('Image preview of the page'),
            'type' => 'string',
            'context' => array('list', 'single'),
        );
        $schema['properties']['icon'] = array(
            'description' => __('Icon of the page'),
            'type' => 'string',
            'context' => array('list', 'single'),
        );
        return $schema;
    }

}
