<?php
/**
 * REST API: EE_REST_Posts_Controller class
 *
 * @package Easyeco New
 * @since 1.0.0
 */

/**
 * Class to access posts via the REST API.
 *
 * @since 1.0.0
 *
 * @see WP_REST_Posts_Controller
 */
class EE_REST_Posts_Controller extends WP_REST_Posts_Controller
{
    /**
     * Post type.
     *
     * @since 1.0.0
     * @var string
     */
    protected $post_type;

    /**
     * Instance of a post meta fields object.
     *
     * @since 1.0.0
     * @var WP_REST_Post_Meta_Fields
     */
    protected $meta;

    /**
     * Constructor.
     *
     * @since 1.0.0
     *
     * @param string $post_type Post type.
     */
    public function __construct()
    {
        parent::__construct('post');
        $this->namespace = 'ee/v1';
    }

    /**
     * Registers the routes for the objects of the controller.
     *
     * @since 1.0.0
     *
     * @see register_rest_route()
     */
    public function register_routes()
    {

        register_rest_route(
            $this->namespace,
            '/' . $this->rest_base,
            array(
                array(
                    'methods' => WP_REST_Server::READABLE,
                    'callback' => array($this, 'get_items'),
                    'permission_callback' => array($this, 'get_items_permissions_check'),
                    'args' => $this->get_collection_params(),
                ),
                'schema' => array($this, 'get_public_item_schema'),
            )
        );

        register_rest_route(
            $this->namespace,
            '/' . $this->rest_base . '/(?P<id>[\d]+)',
            array(
                'args' => array(
                    'id' => array(
                        'description' => __('Unique identifier for the object.'),
                        'type' => 'integer',
                    ),
                ),
                array(
                    'methods' => WP_REST_Server::READABLE,
                    'callback' => array($this, 'get_item'),
                    'permission_callback' => array($this, 'get_item_permissions_check'),
                ),
                'schema' => array($this, 'get_public_item_schema'),
            )
        );
    }
    /**
     * Retrieves the query params for the posts collection.
     *
     * @since 1.0.0
     *
     * @return array Collection parameters.
     */
    public function get_collection_params()
    {
        $query_params = parent::get_collection_params();
        $query_params['per_page'] = array(
            'description'       => __( 'Maximum number of items to be returned in result set.' ),
            'type'              => 'integer',
            'default'           => 25,
            'minimum'           => 1,
            'maximum'           => 100,
            'sanitize_callback' => 'absint',
            'validate_callback' => 'rest_validate_request_arg',
        );
        return $query_params;
    }
    /**
     * Retrieves a collection of posts.
     *
     * @since 1.0.0
     *
     * @param WP_REST_Request $request Full details about the request.
     * @return WP_REST_Response|WP_Error Response object on success, or WP_Error object on failure.
     */
    public function get_items($request)
    {
        $zone_id = $request->get_header('zone-id');

        if (!$zone_id) {
            return new WP_Error(
                'rest_no_zone_id_header_defined',
                __('You need to define a zone id header.'),
                array('status' => 400)
            );
        }

        // Ensure a search string is set in case the orderby is set to 'relevance'.
        if (!empty($request['orderby']) && 'relevance' === $request['orderby'] && empty($request['search'])) {
            return new WP_Error(
                'rest_no_search_term_defined',
                __('You need to define a search term to order by relevance.'),
                array('status' => 400)
            );
        }

        // Ensure an include parameter is set in case the orderby is set to 'include'.
        if (!empty($request['orderby']) && 'include' === $request['orderby'] && empty($request['include'])) {
            return new WP_Error(
                'rest_orderby_include_missing_include',
                __('You need to define an include parameter to order by include.'),
                array('status' => 400)
            );
        }

        // Retrieve the list of registered collection query parameters.
        $registered = $this->get_collection_params();
        $args = array();

        /*
         * This array defines mappings between public API query parameters whose
         * values are accepted as-passed, and their internal WP_Query parameter
         * name equivalents (some are the same). Only values which are also
         * present in $registered will be set.
         */
        $parameter_mappings = array(
            'author' => 'author__in',
            'author_exclude' => 'author__not_in',
            'exclude' => 'post__not_in',
            'include' => 'post__in',
            'menu_order' => 'menu_order',
            'offset' => 'offset',
            'order' => 'order',
            'orderby' => 'orderby',
            'page' => 'paged',
            'parent' => 'post_parent__in',
            'parent_exclude' => 'post_parent__not_in',
            'search' => 's',
            'slug' => 'post_name__in',
            'status' => 'post_status',
        );

        /*
         * For each known parameter which is both registered and present in the request,
         * set the parameter's value on the query $args.
         */
        foreach ($parameter_mappings as $api_param => $wp_param) {
            if (isset($registered[$api_param], $request[$api_param])) {
                $args[$wp_param] = $request[$api_param];
            }
        }

        // Check for & assign any parameters which require special handling or setting.
        $args['date_query'] = array();

        // Set before into date query. Date query must be specified as an array of an array.
        if (isset($registered['before'], $request['before'])) {
            $args['date_query'][0]['before'] = $request['before'];
        }

        // Set after into date query. Date query must be specified as an array of an array.
        if (isset($registered['after'], $request['after'])) {
            $args['date_query'][0]['after'] = $request['after'];
        }

        // Ensure our per_page parameter overrides any provided posts_per_page filter.
        if (isset($registered['per_page'])) {
            $args['posts_per_page'] = $request['per_page'];
        }

        if (isset($registered['sticky'], $request['sticky'])) {
            $sticky_posts = get_option('sticky_posts', array());
            if (!is_array($sticky_posts)) {
                $sticky_posts = array();
            }
            if ($request['sticky']) {
                /*
                 * As post__in will be used to only get sticky posts,
                 * we have to support the case where post__in was already
                 * specified.
                 */
                $args['post__in'] = $args['post__in'] ? array_intersect($sticky_posts, $args['post__in']) : $sticky_posts;

                /*
                 * If we intersected, but there are no post ids in common,
                 * WP_Query won't return "no posts" for post__in = array()
                 * so we have to fake it a bit.
                 */
                if (!$args['post__in']) {
                    $args['post__in'] = array(0);
                }
            } elseif ($sticky_posts) {
                /*
                 * As post___not_in will be used to only get posts that
                 * are not sticky, we have to support the case where post__not_in
                 * was already specified.
                 */
                $args['post__not_in'] = array_merge($args['post__not_in'], $sticky_posts);
            }
        }

        // Force the post_type argument, since it's not a user input variable.
        $args['post_type'] = $this->post_type;

        /**
         * Filters the query arguments for a request.
         *
         * Enables adding extra arguments or setting defaults for a post collection request.
         *
         * @since 4.7.0
         *
         * @link https://developer.wordpress.org/reference/classes/wp_query/
         *
         * @param array           $args    Key value array of query var to query value.
         * @param WP_REST_Request $request The request used.
         */
        $args = apply_filters("rest_{$this->post_type}_query", $args, $request);
        $query_args = $this->prepare_items_query($args, $request);

        $taxonomies = wp_list_filter(get_object_taxonomies($this->post_type, 'objects'), array('show_in_rest' => true));

        if (!empty($request['tax_relation'])) {
            $query_args['tax_query'] = array('relation' => $request['tax_relation']);
        }

        foreach ($taxonomies as $taxonomy) {
            $base = !empty($taxonomy->rest_base) ? $taxonomy->rest_base : $taxonomy->name;
            $tax_exclude = $base . '_exclude';

            if (!empty($request[$base])) {
                $query_args['tax_query'][] = array(
                    'taxonomy' => $taxonomy->name,
                    'field' => 'term_id',
                    'terms' => $request[$base],
                    'include_children' => false,
                );
            }

            if (!empty($request[$tax_exclude])) {
                $query_args['tax_query'][] = array(
                    'taxonomy' => $taxonomy->name,
                    'field' => 'term_id',
                    'terms' => $request[$tax_exclude],
                    'include_children' => false,
                    'operator' => 'NOT IN',
                );
            }
        }

        $query_args['tax_query'][] = array(
            'taxonomy' => 'municipality',
            'field' => 'term_id',
            'terms' => $zone_id,
            'include_children' => false,
        );

        $posts_query = new WP_Query();
        $query_result = $posts_query->query($query_args);

        // Allow access to all password protected posts if the context is edit.
        if ('edit' === $request['context']) {
            add_filter('post_password_required', '__return_false');
        }

        $posts = array();

        foreach ($query_result as $post) {
            if (!$this->check_read_permission($post)) {
                continue;
            }

            $data = $this->prepare_item_for_response($post, $request);
            $posts[] = $this->prepare_response_for_collection($data);
        }

        // Reset filter.
        if ('edit' === $request['context']) {
            remove_filter('post_password_required', '__return_false');
        }

        $page = (int) $query_args['paged'];
        $total_posts = $posts_query->found_posts;

        if ($total_posts < 1) {
            // Out-of-bounds, run the query again without LIMIT for total count.
            unset($query_args['paged']);

            $count_query = new WP_Query();
            $count_query->query($query_args);
            $total_posts = $count_query->found_posts;
        }

        $max_pages = ceil($total_posts / (int) $posts_query->query_vars['posts_per_page']);

        if ($page > $max_pages && $total_posts > 0) {
            return new WP_Error(
                'rest_post_invalid_page_number',
                __('The page number requested is larger than the number of pages available.'),
                array('status' => 400)
            );
        }

        $response = rest_ensure_response($posts);

        $response->header('X-WP-Total', (int) $total_posts);
        $response->header('X-WP-TotalPages', (int) $max_pages);

        $request_params = $request->get_query_params();
        $base = add_query_arg(urlencode_deep($request_params), rest_url(sprintf('%s/%s', $this->namespace, $this->rest_base)));

        if ($page > 1) {
            $prev_page = $page - 1;

            if ($prev_page > $max_pages) {
                $prev_page = $max_pages;
            }

            $prev_link = add_query_arg('page', $prev_page, $base);
            $response->link_header('prev', $prev_link);
        }
        if ($max_pages > $page) {
            $next_page = $page + 1;
            $next_link = add_query_arg('page', $next_page, $base);

            $response->link_header('next', $next_link);
        }

        return $response;
    }

    /**
     * Retrieves a single post.
     *
     * @since 1.0.0
     *
     * @param WP_REST_Request $request Full details about the request.
     * @return WP_REST_Response|WP_Error Response object on success, or WP_Error object on failure.
     */
    public function get_item($request)
    {
        $post = $this->get_post($request['id']);
        if (is_wp_error($post)) {
            return $post;
        }

        $data = $this->prepare_item_for_response($post, $request);
        $response = rest_ensure_response($data);

        if (is_post_type_viewable(get_post_type_object($post->post_type))) {
            $response->link_header('alternate', get_permalink($post->ID), array('type' => 'text/html'));
        }

        return $response;
    }

    /**
     * Prepares a single post output for response.
     *
     * @since 1.0.0
     *
     * @param WP_Post         $post    Post object.
     * @param WP_REST_Request $request Request object.
     * @return WP_REST_Response Response object.
     */
    public function prepare_item_for_response($post, $request)
    {
        $GLOBALS['post'] = $post;

        setup_postdata($post);

        // Base fields for every post.
        $data = array();

        $data['id'] = $post->ID;

        $data['slug'] = $post->post_name;

        $data['title'] = array();

        add_filter('protected_title_format', array($this, 'protected_title_format'));
        $data['title']['rendered'] = get_the_title($post->ID);
        remove_filter('protected_title_format', array($this, 'protected_title_format'));

        $data['date'] = $this->prepare_date_response($post->post_date_gmt, $post->post_date);

        if ('0000-00-00 00:00:00' === $post->post_date_gmt) {
            $post_date_gmt = get_gmt_from_date($post->post_date);
        } else {
            $post_date_gmt = $post->post_date_gmt;
        }
        $data['date_gmt'] = $this->prepare_date_response($post_date_gmt);

        $data['link'] = get_permalink($post->ID);

        $data['content'] = array();
        $data['content']['rendered'] = post_password_required($post) ? '' : apply_filters('the_content', $post->post_content);

        $excerpt = apply_filters('get_the_excerpt', $post->post_excerpt, $post);
        $excerpt = apply_filters('the_excerpt', $excerpt);

        $data['excerpt'] = array(
            'rendered' => post_password_required($post) ? '' : $excerpt,
        );

        $data['featured_media_url'] = get_the_post_thumbnail_url($post->ID);

        $data['featured_media'] = (int) get_post_thumbnail_id($post->ID);

        $data['sticky'] = is_sticky($post->ID);

        $template = get_page_template_slug($post->ID);
        if ($template) {
            $data['template'] = $template;
        } else {
            $data['template'] = '';
        }

        $data['format'] = get_post_format($post->ID);

        // Fill in blank post format.
        if (empty($data['format'])) {
            $data['format'] = 'standard';
        }

        $data['meta'] = $this->meta->get_value($post->ID, $request);

        $context = !empty($request['context']) ? $request['context'] : 'view';

        $data = $this->filter_response_by_context($data, $context);

        // Wrap the data in a response object.
        $response = rest_ensure_response($data);

        /**
         * Filters the post data for a response.
         *
         * The dynamic portion of the hook name, `$this->post_type`, refers to the post type slug.
         *
         * @since 1.0.0
         *
         * @param WP_REST_Response $response The response object.
         * @param WP_Post          $post     Post object.
         * @param WP_REST_Request  $request  Request object.
         */
        return apply_filters("rest_prepare_{$this->post_type}", $response, $post, $request);
    }

}
