<?php
/**
 * REST API: EE_REST_Reports_Controller class
 *
 * @package Easyeco New
 * @since 1.0.0
 */

/**
 * Class to access reports via the REST API.
 *
 * @since 1.0.0
 *
 * @see WP_REST_Posts_Controller
 */
class EE_REST_Reports_Controller extends WP_REST_Posts_Controller
{
    /**
     * Post type.
     *
     * @since 1.0.0
     * @var string
     */
    protected $post_type;

    /**
     * Instance of a post meta fields object.
     *
     * @since 1.0.0
     * @var WP_REST_Post_Meta_Fields
     */
    protected $meta;

    /**
     * Constructor.
     *
     * @since 1.0.0
     *
     * @param string $post_type Post type.
     */
    public function __construct()
    {
        parent::__construct('report');
        $this->namespace = 'ee/v1';
    }

    /**
     * Registers the routes for the objects of the controller.
     *
     * @since 1.0.0
     *
     * @see register_rest_route()
     */
    public function register_routes()
    {
        register_rest_route(
            $this->namespace,
            '/' . $this->rest_base,
            array(
                array(
                    'methods' => WP_REST_Server::CREATABLE,
                    'callback' => array($this, 'create_item'),
                    //'permission_callback' => array($this, 'create_item_permissions_check'),
                    'args' => $this->get_endpoint_args_for_item_schema(WP_REST_Server::CREATABLE),
                ),
                'schema' => array($this, 'get_public_item_schema'),
            )
        );
    }

    /**
     * Retrieves the post's schema, conforming to JSON Schema.
     *
     * @since 1.0.0
     *
     * @return array Item schema data.
     */
    public function get_item_schema()
    {
        if ($this->schema) {
            return $this->add_additional_fields_schema($this->schema);
        }

        $schema = array(
            '$schema' => 'http://json-schema.org/draft-04/schema#',
            'title' => $this->post_type,
            'type' => 'object',
            // Base properties for every Post.
            'properties' => array(
                'type' => array(
                    'description' => __("The type of the report information"),
                    'type' => 'string',
                    'enum' => array('info', 'report', 'claim', 'withdrawal'),
                    'context' => array('edit', 'view'),
                    'required' => true,
                ),
                'status' => array(
                    'description' => __("The status of the report"),
                    'type' => 'string',
                    'enum' => array('not_taken', 'taken', 'canceled', 'closed'),
                    'context' => array('edit', 'view'),
                    'required' => false,
                ),
                'firstname' => array(
                    'description' => __('Firstname of the person who is writing the report'),
                    'type' => 'string',
                    'context' => array('edit', 'view'),
                    'required' => true,
                ),
                'lastname' => array(
                    'description' => __('Lastname of the person who is writing the report'),
                    'type' => 'string',
                    'context' => array('edit', 'view'),
                    'required' => true,
                ),
                'cf' => array(
                    'description' => __('Codice Fiscale'),
                    'type' => 'string',
                    'context' => array('edit', 'view'),
                ),
                'address' => array(
                    'description' => __('Address of the person who is writing the report'),
                    'type' => 'string',
                    'context' => array('edit', 'view'),
                    'required' => true,
                ),
                'phone' => array(
                    'description' => __('Phone of the person who is writing the report'),
                    'type' => 'string',
                    'context' => array('edit', 'view'),
                    'required' => true,
                ),
                'email' => array(
                    'description' => __('Email of the person who is writing the report'),
                    'type' => 'string',
                    'context' => array('edit', 'view'),
                    'required' => true,
                ),
                'garbage_type' => array(
                    'description' => __('type of garbage for withdrawal'),
                    'type' => 'string',
                    'context' => array('edit', 'view'),
                ),
                'garbage_quantity' => array(
                    'description' => __('quantity of garbage for withdrawal'),
                    'type' => 'number',
                    'context' => array('edit', 'view'),
                ),
                'note' => array(
                    'description' => __('Note of the person who is writing the report'),
                    'type' => 'string',
                    'context' => array('edit', 'view'),
                ),
                'lat' => array(
                    'description' => __('Latitude of the person who is writing the report'),
                    'type' => 'number',
                    'context' => array('edit', 'view'),
                ),
                'lng' => array(
                    'description' => __('Longitude of the person who is writing the report'),
                    'type' => 'number',
                    'context' => array('edit', 'view'),
                ),
                'images' => array(
                    'description' => __('Images for the report'),
                    'type' => 'array',
                    'items' => array(
                        'type' => 'files',
                    ),
                    'context' => array('edit'),
                ),
            ),
        );

        // Take a snapshot of which fields are in the schema pre-filtering.
        $schema_fields = array_keys($schema['properties']);

        /**
         * Filter the post's schema.
         *
         * The dynamic portion of the filter, `$this->post_type`, refers to the
         * post type slug for the controller.
         *
         * @since 5.4.0
         *
         * @param array $schema Item schema data.
         */
        $schema = apply_filters("rest_{$this->post_type}_item_schema", $schema);

        // Emit a _doing_it_wrong warning if user tries to add new properties using this filter.
        $new_fields = array_diff(array_keys($schema['properties']), $schema_fields);
        if (count($new_fields) > 0) {
            _doing_it_wrong(__METHOD__, __('Please use register_rest_field to add new schema properties.'), '5.4.0');
        }

        $this->schema = $schema;

        return $this->add_additional_fields_schema($this->schema);
    }

    /**
     * Creates a single post.
     *
     * @since 1.0.0
     *
     * @param WP_REST_Request $request Full details about the request.
     * @return WP_REST_Response|WP_Error Response object on success, or WP_Error object on failure.
     */
    public function create_item($request)
    {
        $zone_id = $request->get_header('zone-id');
        
        // Ensure a search string is set in case the orderby is set to 'relevance'.
        if (!$zone_id) {
            return new WP_Error(
                'rest_no_zone_id_header_defined',
                __('You need to define a zone id header.'),
                array('status' => 400)
            );
        }
        $zone = get_term($zone_id, 'municipality');
        if (is_wp_error($zone) || is_null($zone)){
            return new WP_Error(
                'rest_zone_by_id_not_found',
                __('No zone was found for the provided id.'),
                array('status' => 400)
            );
        }

        $prepared_post = $this->prepare_item_for_database($request);

        if (is_wp_error($prepared_post)) {
            return $prepared_post;
        }

        $prepared_post->post_type = $this->post_type;
        $prepared_post->post_title = str_replace('{{zone}}', $zone->name, $prepared_post->post_title);

        $post_id = wp_insert_post(wp_slash((array) $prepared_post), true);

        if (is_wp_error($post_id)) {

            if ('db_insert_error' === $post_id->get_error_code()) {
                $post_id->add_data(array('status' => 500));
            } else {
                $post_id->add_data(array('status' => 400));
            }

            return $post_id;
        }

        $term_taxonomy_ids = wp_set_object_terms($post_id, (int)$zone_id, 'municipality');

        $post = get_post($post_id);

        /**
         * Fires after a single post is created or updated via the REST API.
         *
         * The dynamic portion of the hook name, `$this->post_type`, refers to the post type slug.
         *
         * @since 1.0.0
         *
         * @param WP_Post         $post     Inserted or updated post object.
         * @param WP_REST_Request $request  Request object.
         * @param bool            $creating True when creating a post, false when updating.
         */
        do_action("rest_insert_{$this->post_type}", $post, $request, true);

        $post = get_post($post_id);
        $fields_update = $this->update_additional_fields_for_object($post, $request);

        if (is_wp_error($fields_update)) {
            return $fields_update;
        }

        $request->set_param('context', 'edit');

        /**
         * Fires after a single post is completely created or updated via the REST API.
         *
         * The dynamic portion of the hook name, `$this->post_type`, refers to the post type slug.
         *
         * @since 5.0.0
         *
         * @param WP_Post         $post     Inserted or updated post object.
         * @param WP_REST_Request $request  Request object.
         * @param bool            $creating True when creating a post, false when updating.
         */
        do_action("rest_after_insert_{$this->post_type}", $post, $request, true);

        $response = $this->prepare_item_for_response($post, $request);
        $response = rest_ensure_response($response);

        $response->set_status(201);
        $response->header('Location', rest_url(sprintf('%s/%s/%d', $this->namespace, $this->rest_base, $post_id)));

        return $response;
    }

    /**
     * Prepares a single post for create or update.
     *
     * @since 1.0.0
     *
     * @param WP_REST_Request $request Request object.
     * @return stdClass|WP_Error Post object or WP_Error.
     */
    protected function prepare_item_for_database($request)
    {
        $prepared_post = new stdClass();

        $schema = $this->get_item_schema();

        $meta_input = array();
        // Post type.
        if (!empty($schema['properties']['type']) && isset($request['type'])) {
            $meta_input['type'] = $request['type'];
        }

        //Post status
        $meta_input['status'] = 'not_taken';
        $meta_input['not_taken_timestamp'] = time();

        // Post firstname.
        if (!empty($schema['properties']['firstname']) && isset($request['firstname'])) {
            $meta_input['firstname'] = $request['firstname'];
        }

        // Post lastname.
        if (!empty($schema['properties']['lastname']) && isset($request['lastname'])) {
            $meta_input['lastname'] = $request['lastname'];
        }

        // Post CF.
        if (!empty($schema['properties']['cf']) && isset($request['cf'])) {
            $meta_input['cf'] = $request['cf'];
        }

        // Post address.
        if (!empty($schema['properties']['address']) && isset($request['address'])) {
            $meta_input['address'] = $request['address'];
        }

        // Post phone.
        if (!empty($schema['properties']['phone']) && isset($request['phone'])) {
            $meta_input['phone'] = $request['phone'];
        }

        // Post email.
        if (!empty($schema['properties']['email']) && isset($request['email'])) {
            $meta_input['email'] = $request['email'];
        }

        // Post garbage type.
        if (!empty($schema['properties']['garbage_type']) && isset($request['garbage_type'])) {
            $meta_input['garbage_type'] = $request['garbage_type'];
        }

        // Post garbage quantity.
        if (!empty($schema['properties']['garbage_quantity']) && isset($request['garbage_quantity'])) {
            $meta_input['garbage_quantity'] = $request['garbage_quantity'];
        }

        // Post note.
        if (!empty($schema['properties']['note']) && isset($request['note'])) {
            $meta_input['note'] = $request['note'];
        }

        // Post latitude.
        if (!empty($schema['properties']['lat']) && isset($request['lat'])) {
            $meta_input['latitude'] = $request['lat'];
        }

        // Post longitude.
        if (!empty($schema['properties']['lng']) && isset($request['lng'])) {
            $meta_input['longitude'] = $request['lng'];
        }
        if(!empty($schema['properties']['images'])){
            $meta_input['images'] = $this->set_images_for_api();
        }
        $day = date_i18n("d F Y", strtotime());

        if(function_exists('acf_get_field')){
            $type_field = acf_get_field('type');
            $type = $type_field['choices'][$request['type']];
        } else {
            $type = $request['type'];
        }

        $prepared_post->post_title = $type.' - '.$request['firstname'].' '.$request['lastname'].' - {{zone}} - '.$day;
        $prepared_post->meta_input = $meta_input;
        $prepared_post->post_status = 'publish';
        $prepared_post->post_type = 'report';

        /**
         * Filters a post before it is inserted via the REST API.
         *
         * The dynamic portion of the hook name, `$this->post_type`, refers to the post type slug.
         *
         * @since 1.0.0
         *
         * @param stdClass        $prepared_post An object representing a single post prepared
         *                                       for inserting or updating the database.
         * @param WP_REST_Request $request       Request object.
         */
        return apply_filters("rest_pre_insert_{$this->post_type}", $prepared_post, $request);

    }

    private function set_images_for_api()
    {
        $image_ids = array();
        
        if (isset($_FILES['images']) && isset($_FILES['images']['name']) && $_FILES['images']['name'][0] != "") {
            $length = sizeof($_FILES['images']['name']);

            for ($i = 0; $i < $length; $i++) {
                $file = wp_upload_bits($_FILES['images']['name'][$i], null, file_get_contents($_FILES['images']['tmp_name'][$i]));
                $file['name'] = $_FILES['images']['name'][$i];

                if (!$file['error']) {
                    $arg = array(
                        'guid' => $file['url'],
                        'post_title' => $file['name'],
                        'post_content'   => $file['name'],
                        'post_mime_type' => $file['type'],
                        'post_status' => 'inherit',
                    );
                    $attachmentId = wp_insert_attachment($arg, $file['file'], 0);
                    if ($attachmentId && !is_wp_error($attachmentId)) {
                        // Include image.php
                        require_once ABSPATH . 'wp-admin/includes/image.php';
                        
                        // Define attachment metadata
                        $attach_data = wp_generate_attachment_metadata($attachmentId, $file['file']);
                        // Assign metadata to attachment
                        wp_update_attachment_metadata($attachmentId, $attach_data);
                        
                        array_push($image_ids, $attachmentId);
                    }
                }
            }
        }

        return $image_ids;
    }

    /**
     * Prepares a single post output for response.
     *
     * @since 1.0.0
     *
     * @param WP_Post         $post    Post object.
     * @param WP_REST_Request $request Request object.
     * @return WP_REST_Response Response object.
     */
    public function prepare_item_for_response($post, $request)
    {
        $GLOBALS['post'] = $post;

        setup_postdata($post);

        // Base fields for every post.
        $data = array();

        $data['id'] = $post->ID;

        $context = !empty($request['context']) ? $request['context'] : 'view';
        $data = $this->filter_response_by_context($data, $context);

        // Wrap the data in a response object.
        $response = rest_ensure_response($data);

        /**
         * Filters the post data for a response.
         *
         * The dynamic portion of the hook name, `$this->post_type`, refers to the post type slug.
         *
         * @since 1.0.0
         *
         * @param WP_REST_Response $response The response object.
         * @param WP_Post          $post     Post object.
         * @param WP_REST_Request  $request  Request object.
         */
        return apply_filters("rest_prepare_{$this->post_type}", $response, $post, $request);
    }

}
