<?php

/**
 * The admin-specific functionality of the plugin.
 *
 * @link       https://rextheme.com
 * @since      1.0.0
 *
 * @package    Rex_Product_Metabox
 * @subpackage Rex_Product_Feed/admin
 */


class Rex_Product_Feed_Pro_Ajax {

    /**
     * Hook in ajax handlers.
     *
     * @since    1.0.0
     */
    public static function init() {
        $validations = array(
            'logged_in' => true,
            'user_can'  => 'manage_options',
        );

        // Add custom field to product.
        wp_ajax_helper()->handle( 'rex-product-custom-field' )
                        ->with_callback( array( 'Rex_Product_Feed_Pro_Ajax', 'rex_product_custom_field' ) )
                        ->with_validation( $validations );

        // Add custom product detailed attributes field to product.
        wp_ajax_helper()->handle( 'rex-product-pa-field' )
                        ->with_callback( array( 'Rex_Product_Feed_Pro_Ajax', 'rex_product_pa_field' ) )
                        ->with_validation( $validations );

        // Add Structured data to product.
        wp_ajax_helper()->handle( 'rex-product-structured-data' )
                        ->with_callback( array( 'Rex_Product_Feed_Pro_Ajax', 'rex_product_structured_data' ) )
                        ->with_validation( $validations );

        // Exclude tax to product.
        wp_ajax_helper()->handle( 'rex-product-exclude-tax' )
                        ->with_callback( array( 'Rex_Product_Feed_Pro_Ajax', 'rex_product_exclude_tax' ) )
                        ->with_validation( $validations );

        // Fetch eBay seller category.
        wp_ajax_helper()->handle( 'rex-wpfm-fetch-ebay-category' )
                        ->with_callback( array( 'Rex_Product_Feed_Pro_Ajax', 'rex_product_ebay_category' ) )
                        ->with_validation( $validations );

        wp_ajax_helper()->handle( 'wpfm-enable-drm-pixel' )
                        ->with_callback( array( 'Rex_Product_Feed_Pro_Ajax', 'wpfm_enable_drm_pixel' ) )
                        ->with_validation( $validations );

        wp_ajax_helper()->handle( 'save-drm-pixel-value' )
                        ->with_callback( array( 'Rex_Product_Feed_Pro_Ajax', 'save_drm_pixel_value' ) )
                        ->with_validation( $validations );

        wp_ajax_helper()->handle( 'rex-amazon-fields' )
                        ->with_callback( array( 'Rex_Product_Feed_Pro_Ajax', 'amazon_fields' ) )
                        ->with_validation( $validations );

        wp_ajax_helper()->handle( 'rex-feed-save-email' )
                        ->with_callback( array( 'Rex_Product_Feed_Pro_Ajax', 'rex_feed_save_email' ) )
                        ->with_validation( $validations );

        wp_ajax_helper()->handle( 'rex-feed-export-feed' )
                        ->with_callback( array( 'Rex_Product_Feed_Pro_Ajax', 'rex_feed_export_feed' ) )
                        ->with_validation( $validations );

        wp_ajax_helper()->handle( 'rex-feed-save-pro-license' )
                        ->with_callback( array( 'Rex_Product_Feed_Pro_Ajax', 'rex_feed_save_pro_license' ) )
                        ->with_validation( $validations );

        wp_ajax_helper()->handle( 'rex-feed-handle-feed-rules-content' )
                        ->with_callback( array( 'Rex_Product_Feed_Pro_Ajax', 'rex_feed_get_feed_rules_content' ) )
                        ->with_validation( $validations );

        wp_ajax_helper()->handle( 'rexfeed-enable-google-drm-pixel' )
                        ->with_callback( array( 'Rex_Product_Feed_Pro_Ajax', 'enable_google_drm_pixel' ) )
                        ->with_validation( $validations );

        wp_ajax_helper()->handle( 'rexfeed-save-google-drm-pixel-id' )
                        ->with_callback( array( 'Rex_Product_Feed_Pro_Ajax', 'save_google_drm_pixel_id' ) )
                        ->with_validation( $validations );

        wp_ajax_helper()->handle( 'wpfm-enable-tiktok-pixel' )
            ->with_callback( array( 'Rex_Product_Feed_Pro_Ajax', 'enable_tiktok_pixel' ) )
            ->with_validation( $validations );
    }


    /**
     * WPFM custom field
     * @param $payload
     */
    public static function rex_product_custom_field($payload) {
        $custom_field = isset( $payload['custom_field'] ) ? $payload['custom_field'] : '';

        if( $custom_field !== '' ) {
            update_option( 'rex-wpfm-product-custom-field', $custom_field );

            if( $custom_field !== 'yes' ) {
                delete_option( 'wpfm_product_custom_fields_frontend' );
            }
            wp_send_json_success( $custom_field );
        }
        else{
            delete_option( 'rex-wpfm-product-custom-field' );
            delete_option( 'wpfm_product_custom_fields_frontend' );
            wp_send_json_error();
        }
        wp_send_json_error();
    }


    /**
     * WPFM custom field
     * @param $payload
     * @return string
     */
    public static function rex_product_pa_field($payload) {
        update_option('rex-wpfm-product-pa-field', $payload['pa_field']);
        return 'success';
    }


    /**
     * WPFM custom field
     * @param $payload
     * @return string
     */
    public static function rex_product_structured_data($payload) {
        update_option('rex-wpfm-product-structured-data', $payload['structured_data']);
        return 'success';
    }


    /**
     * @param $payload
     * @return string
     */
    public static function rex_product_exclude_tax($payload) {
        update_option('rex-wpfm-product-structured-data-exclude-tax', $payload['exclude_tax']);
        return 'success';
    }


    /**
     * eBay category list
     * @param $payload
     * @return array
     */
    public static function rex_product_ebay_category($payload) {
        $matches = array();
        $s = $payload['term'];
        $file = plugin_dir_url(__FILE__).'partials/ebay-seller-category.txt';
        $matches = array();
        $handle = @fopen($file, "r");
        while (!feof($handle)) {
            $cat = fgets($handle);
            $haystack = strtolower($cat);
            $needle = strtolower($s);
            if (strpos($haystack, $needle) !== FALSE)
                $matches[] = $cat;
        }
        fclose($handle);
        return json_encode($matches);
    }


    /**
     * @param $payload
     * @return array
     */
    public static function wpfm_enable_drm_pixel($payload) {
        if($payload['wpfm_drm_pixel_enabled'] == 'yes') {
            update_option('wpfm_drm_pixel_enabled', 'yes');
            return array(
                'success' => true,
                'data'  => 'enabled'
            );
        }else if ($payload['wpfm_drm_pixel_enabled'] == 'no') {
            update_option('wpfm_drm_pixel_enabled', 'no');
            return array(
                'success' => true,
                'data'  => 'disabled'
            );
        }
    }


    /**
     * @param $payload
     * @return array
     */
    public static function save_drm_pixel_value($payload) {
        update_option('wpfm_drm_pixel_value', $payload);
        return array(
            'success' => true,
        );
    }


    /**
     * amazon fields
     * @param $payload
     * @return array
     */
    public static function amazon_fields($payload) {
        if($payload['enable_amazon_fields'] === 'yes') {
            update_option('wpfm_amazon_fields', 'yes');
            return array(
                'success' => true,
                'data'  => 'enabled'
            );
        }
        else if ($payload['enable_amazon_fields'] === 'no') {
            update_option('wpfm_amazon_fields', 'no');
            return array(
                'success' => true,
                'data'  => 'disabled'
            );
        }
    }


    /**
     * @desc save user/admin email in settings dashboard
     * @since 6.2.1
     * @param $email
     * @return void
     */
    public static function rex_feed_save_email( $email ) {
        update_option('wpfm_user_email', $email );
        wp_send_json_success();
        wp_die();
    }


    /**
     * Save imported feeds from xml
     *
     * @return void
     * @since 6.3.0
     */
	public static function rex_feed_save_import_xml_feed() {
		check_ajax_referer( 'rex-wpfm-pro-ajax', 'security' );

		if ( empty( $_FILES[ 'rex_feed_import_feed' ][ 'error' ] ) ) {
			$path    = wp_upload_dir();
			$baseurl = $path[ 'baseurl' ] ?? '';
			$basepath = $path[ 'basedir' ] ?? '';

			$feed_data = [];
			$xml_file = $_FILES[ 'rex_feed_import_feed' ] ?? '';
			$file_name = 'temp_' . $xml_file[ 'name' ] ?? '';
			$folder_path = trailingslashit( $basepath ) . 'rex-feed/';

			// make directory if not exist
			if ( !file_exists( $folder_path ) ) {
				wp_mkdir_p( $folder_path );
			}

			$file_path = $folder_path . $file_name;
			if ( !file_exists( $file_path ) ) {
				move_uploaded_file( $xml_file[ 'tmp_name' ], $file_path );
			}

			$file_url = trailingslashit( $baseurl ) . 'rex-feed/' . $file_name;

			if ( function_exists( 'simplexml_load_file' ) ) {
				$feed_data = simplexml_load_file( $file_url, 'SimpleXMLElement', LIBXML_NOCDATA );
				$feed_data = !$feed_data ? simplexml_load_file( $file_path, 'SimpleXMLElement', LIBXML_NOCDATA ) : $feed_data;
			}

			if ( file_exists( $file_path ) && !empty( $feed_data ) ) {
				$feed_title = !empty( $feed_data->title ) ? (array)$feed_data->title : [];
				$feed_title = $feed_title[ 0 ] ?? '';
				if ( $feed_title === 'Rex Product Feeds' ) {
					$feeds = isset($feed_data->feed) ? $feed_data->feed : [];
					foreach ($feeds as $feed) {
						$feed_title = $feed->feed_title ?? '';
						$feed_meta = !empty($feed->feed_meta) ? (array)$feed->feed_meta : [];

						$args = [
							'post_author' => get_current_user_id(),
							'post_title' => $feed_title . ' - Imported',
							'post_content' => '',
							'post_type' => 'product-feed',
							'post_status' => 'publish'
						];
						$post_id = wp_insert_post($args);

						foreach ($feed_meta as $meta_key => $meta_value) {
							$unserialized = @unserialize($meta_value);
							if ($unserialized) {
								update_post_meta($post_id, $meta_key, $unserialized);
							} else {
								update_post_meta($post_id, $meta_key, $meta_value);
							}
						}

						if (isset($feed->feed_cat_ids)) {
							wp_set_object_terms($post_id, unserialize($feed->feed_cat_ids), 'product_cat');
						}

						if (isset($feed->feed_tag_ids)) {
							wp_set_object_terms($post_id, unserialize($feed->feed_tag_ids), 'product_tag');
						}
					}

					$response = [
						'message' => 'Successfully fetched!',
						'feeds' => sizeof( $feeds )
					];
				}
				else {
					$response = [ 'message' => 'Invalid file!' ];
				}
				unlink( $file_path );
				wp_send_json_success( $response );
				wp_die();
			}
		}

		wp_send_json_error( [ 'message' => 'Select a valid file' ] );
		wp_die();
	}

	private function move_uploaded_file() {

	}


    /**
     * @desc Export all published feeds
     * @since 6.3.0
     * @return void
     */
    public static function rex_feed_export_feed( $feed_ids ) {
        $all_feed_data = [];
        $xml_path = false;

        if( is_array( $feed_ids ) && !empty( $feed_ids ) ) {
            foreach( $feed_ids as $id ) {
                $current_feed_data = [
                    'feed_title' => get_the_title( $id ),
                    'feed_meta'  => get_post_meta( $id )
                ];
                $feed_cats         = get_the_terms( $id, 'product_cat' );
                $cats              = [];

                if( !empty( $feed_cats ) ) {
                    foreach( $feed_cats as $cat ) {
                        if( isset( $cat->term_id ) ) {
                            $cats[] = $cat->term_id;
                        }
                    }
                    $cats = serialize( $cats ); //phpcs:ignore
                }

                $feed_tags = get_the_terms( $id, 'product_tag' );
                $tags      = [];

                if( !empty( $feed_tags ) ) {
                    foreach( $feed_tags as $tag ) {
                        if( isset( $tag->term_id ) ) {
                            $tags[] = $tag->term_id;
                        }
                    }
                    $tags = serialize( $tags ); //phpcs:ignore
                }
                if( !empty( $cats ) ) {
                    $current_feed_data[ 'feed_cat_ids' ] = $cats;
                }
                if( !empty( $tags ) ) {
                    $current_feed_data[ 'feed_tag_ids' ] = $tags;
                }
                $all_feed_data[] = $current_feed_data;
            }
        }

        if( is_array( $all_feed_data ) && !empty( $all_feed_data ) ) {
            $xml      = self::generate_xml( $all_feed_data );
            $xml_path = self::download_xml( $xml );
        }

        if ( $xml_path ) {
            wp_send_json_success( [ 'status' => true, 'feeds' => sizeof( $feed_ids ), 'feed_path' => $xml_path ] );
            wp_die();
        }
        wp_send_json_error( [ 'status' => false ] );
        wp_die();
    }


    /**
     * @desc Generate xml3
     * @since 6.3.0
     * @param $all_feeds
     * @return bool|string
     */
    private static function generate_xml( $all_feeds ) {
        $xml = new SimpleXMLElement('<rex_feeds/>');
        $xml->addChild('title', 'Rex Product Feeds' );

        foreach ( $all_feeds as $feed ) {
            if ( is_array( $feed ) && !empty( $feed ) ) {
                $feed_item = $xml->addChild( 'feed' );
                foreach ( $feed as $node => $feed_data ) {
                    if ( is_array( $feed_data ) && !empty( $feed_data ) ) {
                        $feed_meta = $feed_item->addChild( 'feed_meta' );
                        foreach ( $feed_data as $meta_key => $meta_value ) {
                            $meta_value = isset( $meta_value[ 0 ] ) ? $meta_value[ 0 ] : '';
                            if ( '' === $meta_value ) {
                                continue;
                            }
                            $feed_meta->addChild( $meta_key, $meta_value );
                        }
                    }
                    else {
                        if ( '' === $feed_data ) {
                            continue;
                        }
                        $feed_item->addChild( $node, $feed_data );
                    }
                }
            }
        }

        return $xml->asXML();
    }


    /**
     * @desc Save the xml file in /uploads/rex-feed/ folder
     * and gets the url to download.
     * @since 6.3.0
     * @param $xml
     * @return false|string
     */
    private static function download_xml( $xml ) {
        $path    = wp_upload_dir();
        $baseurl = $path[ 'baseurl' ];
        $basepath = $path[ 'basedir' ];
        $file_path = trailingslashit( $basepath ) . 'rex-feed/' . get_bloginfo( 'name' ) . '-rex-product-feeds-import.xml';
        $file_url = trailingslashit( $baseurl ) . 'rex-feed/' . get_bloginfo( 'name' ) . '-rex-product-feeds-import.xml';
        return file_put_contents( $file_path, $xml ) ? $file_url : false;
    }


    /**
     * Save new license key on
     * activate/deactivate license key button
     *
     * @param string $license License key.
     *
     * @return false[]|true[]
     * @since 7.2.18
     */
public static function rex_feed_save_pro_license( $license_data ) {
    // Extract and sanitize input
    $license = isset( $license_data['license'] ) ? trim( $license_data['license'] ) : '';
    $action_type = isset( $license_data['action_type'] ) ? trim( $license_data['action_type'] ) : '';
    $security = isset( $license_data['security'] ) ? $license_data['security'] : '';

    // Schedule license check if not already scheduled
    if ( ! wp_next_scheduled( 'rex_wpfm_license_check' ) ) {
        wp_schedule_event( time(), 'monthly', 'rex_wpfm_license_check' );
    }

    // Validate action type
    $valid_actions = [ 'wpfm_pro_license_activate', 'wpfm_pro_license_deactivate' ];
    if ( ! in_array( $action_type, $valid_actions, true ) ) {
        return;
    }

    // Verify nonce
    if ( empty( $security ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $security ) ), 'rex-wpfm-pro-ajax' ) ) {
        wp_send_json_error( [
            'message' => __( 'Invalid nonce. Please refresh and try again.', 'rex-wpfm-pro' ),
        ] );
    }

    // Determine if activation or deactivation
    $is_activation = ( $action_type === 'wpfm_pro_license_activate' );
    $edd_action = $is_activation ? 'activate_license' : 'deactivate_license';
    
    // Prepare API parameters
    $api_params = [
        'edd_action' => $edd_action,
        'license'    => $license,
        'item_id'    => WPFM_SL_ITEM_ID,
        'url'        => home_url(),
        'source'     => 'rextheme',
    ];

    // Determine which license URL to use
    $status_data = wp_remote_get( WPFM_SL_STORE_URL );
    $license_url = ( is_wp_error( $status_data ) || 200 !== wp_remote_retrieve_response_code( $status_data ) )
        ? WPFM_EDD_LICENSE_URL . '?' . http_build_query( $api_params )
        : WPFM_SL_STORE_URL . '?' . http_build_query( $api_params );

    // Call the API
    $response = wp_remote_get( $license_url, [
        'timeout'   => 15,
        'sslverify' => false,
    ] );

    // Handle API errors
    if ( is_wp_error( $response ) || 200 !== wp_remote_retrieve_response_code( $response ) ) {
        $message = ( is_wp_error( $response ) && ! empty( $response->get_error_message() ) )
            ? $response->get_error_message()
            : __( 'An error occurred, please try again.', 'rex-product-feed-pro' );
        
        wp_send_json( [
            'action_type' => $is_activation ? 'activated' : 'deactivated',
            'license_data' => null,
            'message' => $message,
        ] );
    }

    // Decode response
    $license_response = json_decode( wp_remote_retrieve_body( $response ) );

    // Process response based on success status
    if ( false === $license_response->success ) {
        // Error messages mapping
        $error_messages = [
            'expired' => __( 'Your license key has expired.', 'rex-product-feed-pro' ),
            'revoked' => __( 'Your license key has been disabled.', 'rex-product-feed-pro' ),
            'missing' => __( 'Invalid license.', 'rex-product-feed-pro' ),
            'invalid' => __( 'Your license is not active for this URL.', 'rex-product-feed-pro' ),
            'site_inactive' => __( 'Your license is not active for this URL.', 'rex-product-feed-pro' ),
            'item_name_mismatch' => sprintf( 
                __( 'This appears to be an invalid license key for %s.', 'rex-product-feed-pro' ), 
                'Product Feed Manager for WooCommerce' 
            ),
            'no_activations_left' => __( 'Your license key has reached its activation limit.', 'rex-product-feed-pro' ),
        ];

        $message = isset( $error_messages[ $license_response->error ] )
            ? $error_messages[ $license_response->error ]
            : __( 'An error occurred, please try again.', 'rex-product-feed-pro' );
    } else {
        // Success messages
        if ( $is_activation ) {
            $message = ( $license_response->license === 'valid' )
                ? __( 'Your license is successfully activated.', 'rex-product-feed-pro' )
                : __( 'Your license activation failed.', 'rex-product-feed-pro' );
        } else {
            $message = ( $license_response->license === 'deactivated' )
                ? __( 'Your license is successfully deactivated.', 'rex-product-feed-pro' )
                : __( 'Your license deactivation failed.', 'rex-product-feed-pro' );
        }
    }

    // Update options
    update_option( 'wpfm_pro_license_status', $license_response->license );
    update_option( 'wpfm_pro_license_data', serialize( json_decode( json_encode( $license_response ), true ) ) );
    update_option( 'wpfm_pro_license_key', $license );
    
    // Update premium status for activation
    if ( $is_activation && $license_response->license === 'valid' ) {
        update_option( 'wpfm_is_premium', 'yes' );
    } elseif ( ! $is_activation ) {
        update_option( 'wpfm_is_premium', 'no' );
    }

    // Send JSON response
    wp_send_json( [
        'action_type' => $is_activation ? 'activated' : 'deactivated',
        'license_data' => $license_response,
        'message' => $message,
    ] );
}

    /**
     * Get feed rules row content
     *
     * @param $payload
     *
     * @return array
     * @since 6.4.1
     */
    public static function rex_feed_get_feed_rules_content( $payload ) {
        $status = 'added';
        if( 'click' !== $payload[ 'event' ] ) {
            $status = get_post_meta( $payload[ 'feed_id' ], '_rex_feed_feed_rules_button', true ) ?: 'added';
        }

        if( !apply_filters('wpfm_is_premium', false) || 'added' !== $status ) {
            return [ 'status' => false, 'markups' => '' ];
        }
        $feed_filter_rules = get_post_meta( $payload[ 'feed_id' ], '_rex_feed_feed_config_rules', true ) ?: get_post_meta( $payload[ 'feed_id' ], 'rex_feed_feed_config_rules', true );
        $feed_filter_rules = new Rex_Product_Feed_Rules( $feed_filter_rules );
        ob_start();
        include_once plugin_dir_path(__FILE__) . '/partials/rex-product-feed-pro-feed-rules-body.php';
        $markups = ob_get_contents();
        ob_end_clean();
        return [ 'status' => true, 'markups' => $markups ];
    }

    /**
     * Enable facebook pixel tracking
     *
     * @param array $payload Payload.
     *
     * @return array
     *
     * @since 7.4.5
     */
    public static function enable_google_drm_pixel( $payload ) {
        if ( 'yes' === $payload[ 'wpfm_google_drm_pixel_enabled' ] ) {
            update_option( 'wpfm_google_drm_pixel_enabled', 'yes' );
            return array(
                'success' => true,
                'data'    => 'enabled',
            );
        }
        else {
            update_option( 'wpfm_google_drm_pixel_enabled', 'no' );
            return array(
                'success' => true,
                'data'    => 'disabled',
            );
        }
    }

    /**
     * Save Google DRM pixel ID
     *
     * @param array $payload Payload.
     *
     * @return array
     *
     * @since 7.4.5
     */
    public static function save_google_drm_pixel_id( $payload ) {
        update_option( 'wpfm_google_drm_pixel_id', str_replace( 'AW-', '', $payload ) );
        return array(
            'success' => true,
        );
    }

    /**
     * Enable facebook pixel tracking
     *
     * @param array $payload Payload.
     *
     * @return array
     */
    public static function enable_tiktok_pixel( $payload )
    {
        if ( 'yes' === $payload[ 'wpfm_tiktok_pixel_enabled' ] ) {
            update_option( 'wpfm_tiktok_pixel_enabled', 'yes' );
            return array(
                'success' => true, 'data' => 'enabled',
            );
        }
        else {
            update_option( 'wpfm_tiktok_pixel_enabled', 'no' );
            return array(
                'success' => true, 'data' => 'disabled',
            );
        }
    }
}
