<?php

/**
 * Fired during plugin activation
 *
 * @link       https://rextheme.com
 * @since      1.0.0
 *
 * @package    Rex_Product_Feed_Pro
 * @subpackage Rex_Product_Feed_Pro/includes
 */


/**
 * Fired during plugin activation.
 *
 * This class defines all code necessary to run during the plugin's activation.
 *
 * @since      1.0.0
 * @package    Rex_Product_Feed_Pro
 * @subpackage Rex_Product_Feed_Pro/includes
 * @author     RexTheme <support@rextheme.com>
 */
include_once ABSPATH . 'wp-admin/includes/class-wp-upgrader.php';

if ( ! class_exists( 'Silent_Skin' ) ) {
	/**
	 * A quiet upgrader skin, so as not to output anything during plugin installation.
	 *
	 * @since 7.0.0
	 */
	class Silent_Skin extends \WP_Upgrader_Skin {
		public function feedback( $string, ...$args ) { /* Do nothing */ }
		public function header() { /* Do nothing */ }
		public function footer() { /* Do nothing */ }
		public function error( $errors ) { /* Do nothing */ }
	}
}

/**
 * Fired during plugin activation.
 *
 * This class defines all code necessary to run during the plugin's activation.
 *
 * @since      1.0.0
 * @package    Rex_Product_Feed_Pro
 * @subpackage Rex_Product_Feed_Pro/includes
 * @author     RexTheme <support@rextheme.com>
 */
class Rex_Product_Feed_Pro_Activator {

	/**
	 * Activate Rex Product Feed Pro and ensure PFM Free is installed and active.
	 *
	 * @since    1.0.0
	 */
	public static function activate() {
		// Schedule license check
		if ( ! wp_next_scheduled( 'rex_wpfm_license_check' ) ) {
			wp_schedule_event( time(), 'monthly', 'rex_wpfm_license_check' );
		}

		// Check if user has permission to install/activate plugins
		if ( ! current_user_can( 'activate_plugins' ) ) {
			update_option( 'pfmpro_admin_notice_type', 'permission' );
			return;
		}

		// Check if PFM (free) is installed
		if ( ! function_exists( 'get_plugins' ) ) {
			require_once ABSPATH . 'wp-admin/includes/plugin.php';
		}

		$plugin_path = 'best-woocommerce-feed/rex-product-feed.php';
		$installed_plugins = get_plugins();

		if ( ! isset( $installed_plugins[$plugin_path] ) ) {
			// PFM (free) is not installed, install and activate it
			self::install_and_activate_pfm_free();
		} else {
			// PFM (free) is installed, check if it's active
			if ( ! is_plugin_active( $plugin_path ) ) {
				// PFM (free) is installed but not active, activate it
				$activated = activate_plugin( $plugin_path );
				
				if ( is_wp_error( $activated ) ) {
					update_option( 'pfmpro_admin_notice_type', 'error' );
					update_option( 'pfmpro_admin_error_message', $activated->get_error_message() );
				} else {
					// Get installed version
					$plugin_data = get_plugin_data( WP_PLUGIN_DIR . '/' . $plugin_path );
					$version = isset( $plugin_data['Version'] ) ? $plugin_data['Version'] : '';
					update_option( 'pfmpro_admin_notice_type', 'success' );
					update_option( 'pfmpro_free_version', $version );
				}
			}
		}
	}

	/**
	 * Install and activate PFM (free) plugin.
	 *
	 * @since    1.0.0
	 */
	private static function install_and_activate_pfm_free() {
		// Check installation permission
		if ( ! current_user_can( 'install_plugins' ) ) {
			update_option( 'pfmpro_admin_notice_type', 'permission' );
			return;
		}

		include_once ABSPATH . 'wp-admin/includes/class-wp-upgrader.php';
		include_once ABSPATH . 'wp-admin/includes/plugin-install.php';
		include_once ABSPATH . 'wp-admin/includes/file.php';

		$skin = new Silent_Skin();
		$upgrader = new Plugin_Upgrader( $skin );

		$plugin_slug = 'best-woocommerce-feed';
		$download_url = 'https://downloads.wordpress.org/plugin/' . $plugin_slug . '.latest-stable.zip';

		// Check filesystem permissions
		$credentials = request_filesystem_credentials( '', '', false, false, null );
		if ( false === $credentials ) {
			update_option( 'pfmpro_admin_notice_type', 'permission' );
			return;
		}

		if ( ! WP_Filesystem( $credentials ) ) {
			update_option( 'pfmpro_admin_notice_type', 'permission' );
			return;
		}

		// Install the plugin
		$installed = $upgrader->install( $download_url );

		if ( is_wp_error( $installed ) ) {
			update_option( 'pfmpro_admin_notice_type', 'error' );
			update_option( 'pfmpro_admin_error_message', $installed->get_error_message() );
			return;
		}

		if ( ! $installed ) {
			update_option( 'pfmpro_admin_notice_type', 'error' );
			update_option( 'pfmpro_admin_error_message', 'Installation failed for unknown reasons.' );
			return;
		}

		// Activate the plugin
		$plugin_path = $plugin_slug . '/rex-product-feed.php';
		$activated = activate_plugin( $plugin_path );

		if ( is_wp_error( $activated ) ) {
			update_option( 'pfmpro_admin_notice_type', 'error' );
			update_option( 'pfmpro_admin_error_message', $activated->get_error_message() );
			return;
		}

		// Get installed version
		if ( ! function_exists( 'get_plugin_data' ) ) {
			require_once ABSPATH . 'wp-admin/includes/plugin.php';
		}
		
		$plugin_data = get_plugin_data( WP_PLUGIN_DIR . '/' . $plugin_path );
		$version = isset( $plugin_data['Version'] ) ? $plugin_data['Version'] : '';

		// Set success notice with version
		update_option( 'pfmpro_admin_notice_type', 'success' );
		update_option( 'pfmpro_free_version', $version );
	}

	/**
	 * Display admin notices based on activation results.
	 * This should be hooked to 'admin_notices' in your main plugin file.
	 *
	 * @since    1.0.0
	 */
	public static function display_admin_notices() {
		$notice_type = get_option( 'pfmpro_admin_notice_type' );

		if ( ! $notice_type ) {
			return;
		}

		$message = '';
		$class = 'notice ';

		switch ( $notice_type ) {
			case 'success':
				$version = get_option( 'pfmpro_free_version', '' );
				$version_text = $version ? ' version ' . esc_html( $version ) : '';
				$message = 'Best WooCommerce Feed (PFM Free)' . $version_text . ' has been automatically installed and activated to complete your Rex Product Feed Pro setup.';
				$class .= 'notice-success';
				break;

			case 'error':
				$error_message = get_option( 'pfmpro_admin_error_message', '' );
				$message = 'Rex Product Feed Pro requires Best WooCommerce Feed (PFM free version), but automatic installation failed. Please install it manually from the WordPress plugin repository.';
				if ( $error_message ) {
					$message .= ' Error: ' . esc_html( $error_message );
				}
				$class .= 'notice-error';
				break;

			case 'permission':
				$message = "You don't have permission to install plugins. Please contact your site administrator to install Best WooCommerce Feed (PFM free version).";
				$class .= 'notice-warning';
				break;
		}

		if ( $message ) {
			printf(
				'<div class="%s is-dismissible"><p>%s</p></div>',
				esc_attr( $class ),
				wp_kses_post( $message )
			);

			// Clear the notices after displaying
			delete_option( 'pfmpro_admin_notice_type' );
			delete_option( 'pfmpro_admin_error_message' );
			delete_option( 'pfmpro_free_version' );
		}
	}
}