<?php

/**
 * Copyright (c) HeidiPay.
 */

namespace HeidiPay\Classes;

class API {

    /** @var array STATUS */
    const STATUS = [
        'INITIALISED' => 'initialised',
        'ONGOING' => 'application_ongoing',
        'ABANDONED' => 'abandoned',
        'CLOSED' => 'closed',
        'APPROVED' => 'performing',
        'DELAYED' => 'delayed',
        'OVERDUE' => 'overdue',
        'COMPLETED' => 'completed'
    ];

    /** @var string AUTH_URL */
    const AUTH_URL = 'auth/generate/';

    /** @var string INIT_TRANS_URL */
    const INIT_TRANS_URL = 'api/checkout/v1/init/';

    /** @var string CONTRACT_STATUS_URL */
    const CONTRACT_STATUS_URL = 'api/checkout/v1/status/';

    /** @var string CONTRACT_REFUND_URL */
    const CONTRACT_REFUND_URL = 'api/checkout/v1/refund/';

    /** @var string CONTRACT_API_AVAILABLE */
    const CONTRACT_API_AVAILABLE = 'api/checkout/v1/available/';

    /**
     * Get heidi domain
     * @return string
     */
    public static function getHeidiDomain()
    {
        $domain = "https://origination.heidipay.com/";
        $gateways = WC()->payment_gateways->payment_gateways();
        if (empty($gateways['pagolight'])) {
            return $domain;
        }

        /** Get gateway settings */
        $gateway = $gateways['pagolight'];
        $gateway->init_settings();
        $sandbox = $gateway->settings['sandbox'];
        if ($sandbox == 'yes') {
            $domain = "https://sandbox-origination.heidipay.com/";
        }
        return $domain;
    }

    /**
     * Get auth token
     * @param bool $isPro
     * @return string|bool
     */
    public static function getAuthToken(bool $isPro = false)
    {
        $gateways = WC()->payment_gateways->payment_gateways();
        if (empty($gateways['pagolight'])) {
            return false;
        }

        /** Get gateway settings */
        $gateway = $gateways['pagolight'];
        $gateway->init_settings();
        $key = ($isPro ? 'heidi_secret_key_pro' : 'heidi_secret_key');

        $merchantKey = $gateway->settings[$key];
        if (!$merchantKey) {
            return false;
        }
        $body = ['merchant_key' => $merchantKey];
        $url = self::getHeidiDomain() . self::AUTH_URL;
        $request_util = new Requester();
        $request_util->addHeader('X-Client-Module-Version', Helper::getVersion());
        $request_util->setUrl($url);
        $response = $request_util->post($body);

        $response = json_decode($response['contents'], true);

        if (
            empty($response['status'])
            || $response['status'] !== 'success'
            || empty($response['data']['token'])
        ) {
            return false;
        }

        return $response['data']['token'];
    }

    /**
     * Request refund
     * @param string $externalUuid 
     * @param float $amount 
     * @param string $currency
     * @param bool $isPro
     * @return bool
     */
    public static function requestRefund($externalUuid, $amount, $currency, $isPro)
    {
        $authToken = API::getAuthToken($isPro);
        if (!$authToken) {
            return false;
        }

        $url = API::getHeidiDomain() . API::CONTRACT_REFUND_URL;
        $request_util = new Requester();
        $request_util->addHeader('Authorization', 'Token ' . $authToken);
        $request_util->addHeader('X-Client-Module-Version', Helper::getVersion());
        $request_util->setUrl($url);
        $result = $request_util->post([
            'external_uuid' => $externalUuid,
            'amount' => number_format($amount, 2, '.', ''),
            'currency' => $currency
        ]);

        if ((int) $result['code'] !== 200) {
            $errorMessage = 'Refund request failed. Error: ' . $result['code'];
            Helper::logError($errorMessage);
            return false;
        }

        return true;
    }

    /**
     * Init transaction
     * @param array $requestData 
     * @param string $authToken
     * @return array
     */
    public static function initTransaction(array $requestData, $authToken)
    {
        $url = API::getHeidiDomain() . API::INIT_TRANS_URL;
        $request_util = new Requester();
        $request_util->addHeader('Authorization', 'Token ' . $authToken);
        $request_util->addHeader('X-Client-Module-Version', Helper::getVersion());
        $request_util->setUrl($url);
        $response = $request_util->post($requestData);
        if ($response['code'] != 201 || empty($response['contents'])) {
            return $response;
        }
        return json_decode($response['contents'], true);
    }

    /**
     * Check order status
     * @param string $authToken
     * @param string $external_contract_uuid
     * @return bool
     */
    public static function checkOrderStatus($authToken, $external_contract_uuid)
    {
        $url = API::getHeidiDomain() . API::CONTRACT_STATUS_URL . $external_contract_uuid . '/';
        $request_util = new Requester();
        $request_util->addHeader('Authorization', 'Token ' . $authToken);
        $request_util->addHeader('X-Client-Module-Version', Helper::getVersion());
        $request_util->setUrl($url);
        $response = $request_util->get();

        if ($response['code'] >= 300) {
            return false; // Must be 2XX status
        }

        $content = json_decode($response['contents'], true);

        if (isset($content['status']) && strtolower($content['status']) === API::STATUS['APPROVED']) {
            return true;
        }

        return false;
    }
}
