<?php 
/**
 * Copyright (c) HeidiPay.
 */

namespace HeidiPay\Classes;

use Exception;
use HeidiPay\Controllers\Gateway;
use WC_Admin_Settings;
use WP_Term;
use WC_Product;
use WC_Cart;

class Filter {

    /** @var Filter */
	private static $instance;

    /** @var string META_PREFIX */
    const META_PREFIX = 'heidipay_filter';

    /** @var int MAX_INSTALMENTS */
    const MAX_INSTALMENTS = 6;

    /** @var array PRODUCT_TYPES */
    const PRODUCT_TYPES = [['id' => 'PAGOLIGHT', 'name' => 'PagoLight BNPL', 'pro' => false], ['id' => 'PAGOLIGHT_PRO', 'name' => 'PagoLight Pro', 'pro' => true]];

    /**
     * Register post type
     * @return void 
     */
    public static function getInstance()
    {
        if (null === self::$instance) {
			self::$instance = new self();
		}
		return self::$instance;
    }

    /**
     * Constructor.
     * @return void 
     */
    public function __construct()
    {
        add_action('add_meta_boxes', [$this, 'addFields']);
		add_action('save_post', [$this, 'save']);
        add_action('admin_enqueue_scripts', [$this, 'registerAssets']);
    }

    /**
     * Register assets
     * @return void 
     */
    public function registerAssets()
    {
        $pluginDirectory = realpath(dirname(__FILE__).'/../');
        $pluginUrl = plugin_dir_url($pluginDirectory);
        $assetsUrl = $pluginUrl.'assets/';
        wp_register_style(
            'heidipay_filter-styles',
            $assetsUrl.'css/admin.css'
        );
    }
    
    /**
     * Add fields
     * @param string $postType
     * @return void
     */
    public function addFields($postType)
    {
        add_meta_box(
            self::META_PREFIX.'_box',
            __('Opzioni del servizio PagoLight', 'pagolight'),
            [$this, 'renderProductFields'],
            ['product'],
            'side',
            'low'
        );
    }

    /**
     * Render fields
     * @param WP_Post $post
     * @return string
     */
    public function renderProductFields($post)
    {
        wp_nonce_field('heidipay_filters_box', 'heidipay_filters_box_nonce');
        wp_enqueue_style('heidipay_filter-styles');

        /** Display $disableHeidiPay */
        $value = get_post_meta($post->ID, self::META_PREFIX.'_disable', true);
        ?>
        <label for="<?php echo self::META_PREFIX.'_disable'; ?>">
            <?php _e('Spunta per disabilitare', 'pagolight'); ?>
        </label>
        <input type="checkbox" id="<?php echo self::META_PREFIX.'_disable'; ?>" 
            name="<?php echo self::META_PREFIX.'_disable'; ?>" 
            <?php echo (esc_attr($value) == true ? 'checked="checked"' : ''); ?>/>
        <?php

        /** Display $applyToBasket */
        $value = get_post_meta($post->ID, self::META_PREFIX.'_apply_to_basket', true);
        ?>
        <br>
        <label for="<?php echo self::META_PREFIX.'_apply_to_basket'; ?>">
            <?php _e('Applica la regola al carrello misto', 'pagolight'); ?>
        </label>
        <input type="checkbox" id="<?php echo self::META_PREFIX.'_apply_to_basket'; ?>" 
            name="<?php echo self::META_PREFIX.'_apply_to_basket'; ?>" 
            <?php echo (esc_attr($value) == true ? 'checked="checked"' : ''); ?>/>
        <?php

        /** Display $instalments */
        $values = get_post_meta($post->ID, self::META_PREFIX.'_terms', true);
        if (! empty($values)) {
            $values = json_decode($values);
        } else {
            $values = [];
        }
        if (! is_array($values)) {
            $values = [];
        }
        ?>
        <br>
        <label for="<?php echo self::META_PREFIX.'_terms'; ?>">
            <?php _e('Scegli il numero di rate', 'pagolight'); ?>
        </label>
        <div class="instalments-checkboxes">
        <?php
        $options = Gateway::getTermOptions();
        foreach ($options as $key => $name) {
            ?>
            <div class="instalment-checkbox">
                <input type="checkbox" id="<?php echo 'term_'.$key; ?>"
                    value="<?php echo $key; ?>"
                    data-pro="<?php echo (in_array($key, Widget::PRO_TERMS) ? 'true' : 'false'); ?>"
                    name="<?php echo self::META_PREFIX.'_terms[]'; ?>" 
                    <?php echo (in_array($key, $values) ? 'checked="checked"' : ''); ?>/>
                <label for="<?php echo 'term_'.$key; ?>" title="<?php echo $name; ?>">
                    <?php echo $key; ?>
                </label>
            </div>
            <?php
        }
        ?>
        </div>
        <script type="text/javascript">
            var defaultPro = <?php echo (Widget::getGeneralBNPLProduct()['pro'] ? 'true' : 'false'); ?>;
        </script>
        <?php

        if (! Widget::CAN_BE_PRO) {
            return;
        }

        /** Display product */
        $value = get_post_meta($post->ID, self::META_PREFIX.'_product', true);
        ?>
        <br>
        <label for="<?php echo self::META_PREFIX.'_product'; ?>">
            <?php _e('Seleziona prodotto', 'pagolight'); ?>
        </label>
        <select class="heidipay_filter_product_select" name="<?php echo self::META_PREFIX.'_product'; ?>" id="<?php echo self::META_PREFIX.'_product'; ?>">
            <option value="">
                <?php echo _e('Default', 'pagolight'); ?>
            </option>
        <?php
        foreach (Filter::PRODUCT_TYPES as $productType) {
            ?>
            <option
                value="<?php echo $productType['id']; ?>"
                data-pro="<?php echo ($productType['pro'] ? 'true' : 'false'); ?>"
                <?php echo ($value == $productType['id'] ? 'selected="selected"' : ''); ?>>
                <?php echo $productType['name']; ?>
            </option>
            <?php
        }
        ?>
        </select>
        <script type="text/javascript" defer>
            (function ($) {
                $(function () {
                    $(document).on('change', ".heidipay_filter_product_select", function () {
                        var pro = $(this).find('option[value="'+$(this).val()+'"]').data('pro');
                        if (pro === undefined) {
                            pro = defaultPro;
                        }
                        if (pro) {
                            $(this).closest('#heidipay_filter_box').find('.instalment-checkbox input[type="checkbox"][data-pro=true]').closest('.instalment-checkbox').removeClass('hidden');
                        } else {
                            $(this).closest('#heidipay_filter_box').find('.instalment-checkbox input[type="checkbox"][data-pro=true]').prop('checked', false);
                            $(this).closest('#heidipay_filter_box').find('.instalment-checkbox input[type="checkbox"][data-pro=true]').closest('.instalment-checkbox').addClass('hidden');
                        }
                    });
                    $('.heidipay_filter_product_select').trigger('change');
                });
            })(jQuery);
        </script>
        <?php

        /** Display code */
        $value = get_post_meta($post->ID, self::META_PREFIX.'_code', true);
        ?>
        <br>
        <label for="<?php echo self::META_PREFIX.'_code'; ?>">
            <?php _e('Inserisci codice tabella (solo per Pagolight Pro)', 'pagolight'); ?>
        </label>
        <input type="text" id="<?php echo self::META_PREFIX.'_code'; ?>" 
            name="<?php echo self::META_PREFIX.'_code'; ?>" 
            value="<?php echo esc_attr($value); ?>"/>
        <?php

    }

    /**
     * Save
     * @param int $postId 
     * @return void 
     */
    public function save($postId)
    {
        /** Check nonce */
        if (! isset($_POST['heidipay_filters_box_nonce'] ) ) {
			return $postId;
		}
		$nonce = $_POST['heidipay_filters_box_nonce'];
		if (! wp_verify_nonce($nonce, 'heidipay_filters_box')) {
			return $postId;
		}

		/** Check autosave */
		if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
			return $postId;
		}

		/** Check user permissions */
		if (! current_user_can('edit_post', $postId)) {
            return $postId;
        }

		/** Save disable */
        $disable = (! empty($_POST[self::META_PREFIX.'_disable']) && $_POST[self::META_PREFIX.'_disable'] == 'on');
        if (! $disable) {
            delete_post_meta($postId, self::META_PREFIX.'_disable');
        } else {
            update_post_meta($postId, self::META_PREFIX.'_disable', $disable);
        }

        /** Save apply to basket */
        $applyToBasket = (! empty($_POST[self::META_PREFIX.'_apply_to_basket']) && $_POST[self::META_PREFIX.'_apply_to_basket'] == 'on');
        if (! $applyToBasket) {
            delete_post_meta($postId, self::META_PREFIX.'_apply_to_basket');
        } else {
            update_post_meta($postId, self::META_PREFIX.'_apply_to_basket', $applyToBasket);
        }

        /** Save terms */
        $terms = (! empty($_POST[self::META_PREFIX.'_terms']) ? $_POST[self::META_PREFIX.'_terms'] : []);
        if (count($terms) > self::MAX_INSTALMENTS) {
            WC_Admin_Settings::add_error(__('Maximum 6 choices for the number of instalments', 'pagolight'));
            throw new Exception(__('Maximum 6 choices for the number of instalments', 'pagolight'));
        }
        if (empty($terms)) {
            delete_post_meta($postId, self::META_PREFIX.'_terms');
        } else {
            update_post_meta($postId, self::META_PREFIX.'_terms', json_encode($terms));
        }

        /** Save product */
        $product = (! empty($_POST[self::META_PREFIX.'_product']) ? $_POST[self::META_PREFIX.'_product'] : '');
        if (empty($product)) {
            delete_post_meta($postId, self::META_PREFIX.'_product');
        } else {
            update_post_meta($postId, self::META_PREFIX.'_product', $product);
        }

        /** Save code */
        $code = (! empty($_POST[self::META_PREFIX.'_code']) ? $_POST[self::META_PREFIX.'_code'] : '');
        if (empty($code)) {
            delete_post_meta($postId, self::META_PREFIX.'_code');
        } else {
            update_post_meta($postId, self::META_PREFIX.'_code', $code);
        }
    }

    /**
     * Get filter for category
     * @param WP_Term $category 
     * @return false|array
     */
    public static function getFilterForCategory($category)
    {
        if (! is_object($category) || ! $category instanceof WP_Term) {
            return false;
        }

        $meta = get_term_meta($category->term_id);
        $filters = [
            self::META_PREFIX.'_disable' => (! in_array(self::META_PREFIX.'_disable', array_keys($meta)) ? null : reset($meta[self::META_PREFIX.'_disable'])),
            self::META_PREFIX.'_apply_to_basket' => (! in_array(self::META_PREFIX.'_apply_to_basket', array_keys($meta)) ? null : reset($meta[self::META_PREFIX.'_apply_to_basket'])),
            self::META_PREFIX.'_terms' => (! in_array(self::META_PREFIX.'_terms', array_keys($meta)) ? null : reset($meta[self::META_PREFIX.'_terms'])),
            self::META_PREFIX.'_product' => (! in_array(self::META_PREFIX.'_product', array_keys($meta)) ? null : reset($meta[self::META_PREFIX.'_product'])),
            self::META_PREFIX.'_code' => (! in_array(self::META_PREFIX.'_code', array_keys($meta)) ? null : reset($meta[self::META_PREFIX.'_code'])),
        ];

        if ($filters[self::META_PREFIX.'_disable'] == null
            && $filters[self::META_PREFIX.'_apply_to_basket'] == null
            && $filters[self::META_PREFIX.'_terms'] == null
            && $filters[self::META_PREFIX.'_product'] == null
            && $filters[self::META_PREFIX.'_code'] == null
        ) {
            return false;
        }
        
        return $filters;
    }

    /**
     * Get filter for post
     * @param WC_Product $product
     * @return false|array
     */
    public static function getFilterForProduct($product)
    {
        if (! is_object($product) || ! $product instanceof WC_Product ) {
            return false;
        }

        $meta = get_post_meta($product->get_id());
        $filters = [
            self::META_PREFIX.'_disable' => (! in_array(self::META_PREFIX.'_disable', array_keys($meta)) ? null : reset($meta[self::META_PREFIX.'_disable'])),
            self::META_PREFIX.'_apply_to_basket' => (! in_array(self::META_PREFIX.'_apply_to_basket', array_keys($meta)) ? null : reset($meta[self::META_PREFIX.'_apply_to_basket'])),
            self::META_PREFIX.'_terms' => (! in_array(self::META_PREFIX.'_terms', array_keys($meta)) ? null : reset($meta[self::META_PREFIX.'_terms'])),
            self::META_PREFIX.'_product' => (! in_array(self::META_PREFIX.'_product', array_keys($meta)) ? null : reset($meta[self::META_PREFIX.'_product'])),
            self::META_PREFIX.'_code' => (! in_array(self::META_PREFIX.'_code', array_keys($meta)) ? null : reset($meta[self::META_PREFIX.'_code'])),
        ];

        if ($filters[self::META_PREFIX.'_disable'] == null
            && $filters[self::META_PREFIX.'_apply_to_basket'] == null
            && $filters[self::META_PREFIX.'_terms'] == null
            && $filters[self::META_PREFIX.'_product'] == null
            && $filters[self::META_PREFIX.'_code'] == null
        ) {
            return false;
        }
        
        return $filters;
    }
    
    /**
     * Should product display widget
     * @param WC_Product $product 
     * @return bool
     */
    public static function shouldProductDisplayWidget($product)
    {
        /** Check product */
        $disable = get_post_meta($product->get_id(), self::META_PREFIX.'_disable', true);
        if ($disable) {
            return false;
        }

        /** Check categories */
        $categories = get_the_terms($product->get_id(), 'product_cat');
        if (is_array($categories)) {
            foreach ($categories as $category) {
                $disable = get_term_meta($category->term_id, self::META_PREFIX.'_disable', true);
                if ($disable) {
                    return false;
                }
            }
        }

        return true;
    }

    /**
     * Should cart display widget
     * @param WC_Cart $cart 
     * @return bool
     */
    public static function shouldCartDisplayWidget($cart)
    {
        $allDisabled = true;
        $items = $cart->get_cart();
        foreach ($items as $item) {
            $product = $item['data'];

            /** Get default product for variations */
            if (is_a($product, 'WC_Product_Variation')) {
                $parentId = $product->get_parent_id();
                $product = wc_get_product($parentId);
            }
            
            /** Check product */
            $disable = get_post_meta($product->get_id(), self::META_PREFIX.'_disable', true);
            $applyToBasket = get_post_meta($product->get_id(), self::META_PREFIX.'_apply_to_basket', true);
            if ($disable && $applyToBasket) {
                return false;
            }

            /** Check categories */
            $categories = get_the_terms($product->get_id(), 'product_cat');
            if (is_array($categories)) {
                foreach ($categories as $category) {
                    $disable = get_term_meta($category->term_id, self::META_PREFIX.'_disable', true);
                    $applyToBasket = get_term_meta($category->term_id, self::META_PREFIX.'_apply_to_basket', true);
                    if ($disable && $applyToBasket) {
                        return false;
                    }
                }
            }

            /** Check all disabled */
            if ($allDisabled) {
                $shouldDisplay = self::shouldProductDisplayWidget($product);
                if ($shouldDisplay) {
                    $allDisabled = false;
                }
            }
        }
        return ($allDisabled ? false : true);
    }

    /**
     * Get product allowed terms
     * @param WC_Product $product 
     * @param bool $returnFilter
     * @return array|null
     */
    public static function getProductAllowedTerms($product, bool $returnFilter = false)
    {
        /** Get default product for variations */
        if (is_a($product, 'WC_Product_Variation')) {
            $parentId = $product->get_parent_id();
            $product = wc_get_product($parentId);
        }

        /** Check product */
        $terms = get_post_meta($product->get_id(), self::META_PREFIX.'_terms', true);
        if (! empty($terms)) {
            $terms = json_decode($terms);
            if ($returnFilter) {
                return self::getFilterForProduct($product);
            }
            return $terms;
        }

        /** Check categories */
        $categories = get_the_terms($product->get_id(), 'product_cat');
        if (is_array($categories)) {
            foreach ($categories as $category) {
                $terms = get_term_meta($category->term_id, self::META_PREFIX.'_terms', true);
                if (! empty($terms)) {
                    $terms = json_decode($terms);
                    if ($returnFilter) {
                        return self::getFilterForCategory($category);
                    }
                    return $terms;
                }
            }
        }

        return null;
    }

    /**
     * Get cart allowed terms
     * @param WC_Cart $cart 
     * @param bool $returnFilter
     * @return array|null
     */
    public static function getCartAllowedTerms($cart, bool $returnFilter = false)
    {
        $items = $cart->get_cart();
        $minTerms = null;
        $minInstalments = null;
        $allSameTerms = true;
        $lastTerms = null;
        $lastProduct = null;
        $minProduct = null;

        /** Check products */
        foreach ($items as $item) {
            $product = $item['data'];

            /** Get default product for variations */
            if (is_a($product, 'WC_Product_Variation')) {
                $parentId = $product->get_parent_id();
                $product = wc_get_product($parentId);
            }

            $terms = get_post_meta($product->get_id(), self::META_PREFIX.'_terms', true);
            if (! $terms || empty($terms)) {
                $allSameTerms = false;
                continue;
            }
            $terms = json_decode($terms);
            if (! $terms || empty($terms)) {
                $allSameTerms = false;
                continue;
            }
            $applyToBasket = get_post_meta($product->get_id(), self::META_PREFIX.'_apply_to_basket', true);
            if (! empty($terms) && $applyToBasket) {
                if ($minTerms === null || max($terms) < $minTerms) {
                    $minTerms = max($terms);
                    $minInstalments = $terms;
                    $minProduct = $product;
                }
            }
            
            /** Check all same terms */
            if (empty($terms)) {
                $allSameTerms = false;
            } else if ($allSameTerms) {
                if ($lastTerms === null) {
                    $lastTerms = $terms;
                    $lastProduct = $product;
                    continue;
                }
                $intersect = array_intersect($lastTerms, $terms);
                if (count($intersect) !== count($lastTerms)) {
                    $allSameTerms = false;
                }
            }

        }
        if ($allSameTerms && ! empty($lastTerms)) {
            if ($returnFilter) {
                return self::getFilterForProduct($lastProduct);
            }
            return $lastTerms;
        }
        if (! empty($minTerms)) {
            if ($returnFilter) {
                return self::getFilterForProduct($minProduct);
            }
            return $minInstalments;
        }


        $minTerms = null;
        $minInstalments = null;
        $allSameTerms = true;
        $lastTerms = null;
        $lastCategory = null;
        $minCategory = null;

        /** Check categories */
        foreach ($items as $item) {
            $product = $item['data'];
            
            /** Get default product for variations */
            if (is_a($product, 'WC_Product_Variation')) {
                $parentId = $product->get_parent_id();
                $product = wc_get_product($parentId);
            }

            $categories = get_the_terms($product->get_id(), 'product_cat');
            if (! is_array($categories)) {
                continue;
            }
            
            $categoryTerms = null;
            foreach ($categories as $category) {
                $terms = get_term_meta($category->term_id, self::META_PREFIX.'_terms', true);
                if (! $terms || empty($terms)) {
                    continue;
                }
                $terms = json_decode($terms);
                if (! $terms || empty($terms)) {
                    continue;
                }
                $applyToBasket = get_term_meta($category->term_id, self::META_PREFIX.'_apply_to_basket', true);
                if (! empty($terms) && $applyToBasket) {
                    $categoryTerms = $terms;
                    if ($minTerms === null || max($terms) < $minTerms) {
                        $minTerms = max($terms);
                        $minInstalments = $terms;
                        $minCategory = $category;
                    }
                    break;
                }
                
                $categoryTerms = $terms;
                $lastCategory = $category;
            }

            /** Check all same terms */
            if (empty($categoryTerms)) {
                $allSameTerms = false;
            } else if ($allSameTerms && ! empty($categoryTerms)) {
                if ($lastTerms === null) {
                    $lastTerms = $categoryTerms;
                    continue;
                }
                $intersect = array_intersect($lastTerms, $categoryTerms);
                if (count($intersect) !== count($lastTerms)) {
                    $allSameTerms = false;
                }
            }
        }
        if ($allSameTerms && ! empty($lastTerms)) {
            if ($returnFilter) {
                return self::getFilterForCategory($lastCategory);
            }
            return $lastTerms;
        }
        if (! empty($minTerms)) {
            if ($returnFilter) {
                return self::getFilterForCategory($minCategory);
            }
            return $minInstalments;
        }

        return null;
    }
}