<?php
/**
 * Copyright (c) HeidiPay.
 */

namespace HeidiPay\Classes;

use Exception;
use WP_Post;

class Webhook {

    /** @var string META_PREFIX */
    const META_PREFIX = 'heidipay_webhook';

    /** @var string HOOK_ORDER_STATUS */
    const HOOK_ORDER_STATUS = 'order_status';

    /** @var int $orderId */
    public $orderId;

    /** @var string $hook */
    public $hook;

    /** @var string $token */
    public $token;

    /** @var string $key */
    protected $key;

    /**
     * Get token for order
     * @param int $orderId 
     * @param string $hook
     * @return string
     */
    public static function getTokenForOrder(int $orderId, string $hook)
    {
        $webhook = self::getByOrderId($orderId, $hook);
        if (! $webhook) {
            $webhook = self::createForOrder($orderId, $hook);
        }

        if (! empty($webhook->token)) {
            return $webhook->token;
        }
        $webhook->generateToken();
        $webhook->save();
        return $webhook->token;
    }

    /**
     * Get by order ID
     * @param int $orderId
     * @param string $hook
     * @return Webhook|null
     */
    public static function getByOrderId(int $orderId, string $hook)
    {
        $token = get_post_meta($orderId, self::META_PREFIX.'_'.$hook, true);
        if (! $token || empty($token)) {
            return null;
        }
        
        $webhook = new Webhook();
        $webhook->orderId = $orderId;
        $webhook->hook = $hook;
        $webhook->token = $token;
        return $webhook;
    }

    /**
     * Get by token
     * @param string $token 
     * @param string $hook
     * @return Webhook|null
     */
    public static function getByToken(string $token, string $hook)
    {
        $results = get_posts([
            'meta_key' => self::META_PREFIX.'_'.$hook,
            'meta_value' => $token,
            'post_type' => 'shop_order',
            'post_status' => 'any',
            'posts_per_page' => 1
        ]);
        if (! $results || empty($results)) {
            return null;
        }

        /** @var WP_Post $order */
        $order = reset($results);
        $webhook = new Webhook();
        $webhook->orderId = $order->ID;
        $webhook->hook = $hook;
        $webhook->token = $token;
        return $webhook;
    }

    /**
     * Create for order
     * @param int $orderId 
     * @param string $hook
     * @return Webhhok
     */
    protected static function createForOrder(int $orderId, string $hook)
    {
        $webhook = new Webhook();
        $webhook->orderId = $orderId;
        $webhook->hook = $hook;
        $webhook->generateToken();
        $webhook->save();
        return $webhook;
    }

    /**
     * Get random key
     * @return string
     */
    protected static function getRandomKey()
    {
        $characters = '0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ';
        $charactersLength = strlen($characters);
        $randomString = '';
        for ($i = 0; $i < 26; $i++) {
            $randomString .= $characters[rand(0, $charactersLength - 1)];
        }
        return $randomString;
    }

    /**
     * Save
     * @return bool
     */
    protected function save()
    {
        if (empty($this->orderId) || empty($this->hook) || empty($this->token)) {
            return false;
        }
        return (bool) update_post_meta($this->orderId, self::META_PREFIX.'_'.$this->hook, $this->token);
    }

    /**
     * Generate token
     * @return void 
     */
    protected function generateToken()
    {
        if (! empty($this->token)) {
            return;
        }
        if (empty($this->key)) {
            $this->key = self::getRandomKey();
        }
        $passphrase = $this->getPassphrase();
        $this->token = password_hash($passphrase, PASSWORD_DEFAULT);
    }

    /**
     * Get passphrase
     * @return string
     * @throws Exception
     */
    protected function getPassphrase()
    {
        if (empty($this->orderId) || empty($this->hook)) {
            throw new Exception('Order or hook not defined before passphrase.');
        }
        if (empty($this->key)) {
            $this->key = self::getRandomKey();
        }
        return $this->key.':'.$this->orderId.':'.$this->hook;
    }

    /**
     * Execute
     * @return never 
     */
    public function execute($data)
    {
        switch ($this->hook) {
            case self::HOOK_ORDER_STATUS:
                $this->executeOrderStatus($data);
                break;
            default:
                http_response_code(404);
                echo json_encode([
                    'status' => 'ERROR',
                    'message' => 'Hook not found'
                ]);
                die;
                break;
        }
    }

    /**
     * Execute order status
     * @param array $data 
     * @return never
     */
    protected function executeOrderStatus($data)
    {
        $order = wc_get_order($this->orderId);
        if (empty($order) || ! $order) {
            http_response_code(404);
            echo json_encode([
                'status' => 'ERROR',
                'message' => 'Order not found with this ID'
            ]);
            die;
        }

        /** Change order state */
        $newOrderState = false;
        switch ($data['status']) {
            case API::STATUS['APPROVED']:
                /** @see wc_get_order_statuses() */
                $newOrderState = 'processing';
                // $order->payment_complete();
                break;

            case API::STATUS['ABANDONED']:
                /** @see wc_get_order_statuses() */
                $newOrderState = 'cancelled';
                break;

            case API::STATUS['CLOSED']:
            case API::STATUS['INITIALISED']:
            case API::STATUS['ONGOING']:
            case API::STATUS['DELAYED']:
            case API::STATUS['OVERDUE']:
            case API::STATUS['COMPLETED']:
                /** @todo Next versions - Each status with different behaviors? */
                http_response_code(200);
                echo json_encode([
                    'status' => 'OK',
                    'message' => 'Nothing to do with this status'
                ]);
                die;
            default:
                break;
        }

        /** Order state not found */
        if (! $newOrderState) {
            http_response_code(404);
            echo json_encode([
                'status' => 'ERROR',
                'message' => 'Status not found'
            ]);
            die;
        }

        /** Update order status */
        if ($order->get_status() !== $newOrderState) {
            $order->update_status($newOrderState);
        }

        http_response_code(200);
        echo json_encode([
            'status' => 'OK',
            'message' => 'Success'
        ]);
        die;
    }
}