<?php 
/**
 * Copyright (c) HeidiPay.
 */

namespace HeidiPay\Controllers;

use HeidiPay\Classes\Webhook as WebhookObject;

class Webhook {

    /** @var Filter */
	private static $instance;

    /**
     * Register post type
     * @return void 
     */
    public static function getInstance()
    {
        if (null === self::$instance) {
			self::$instance = new self();
		}
		return self::$instance;
    }

    public function __construct()
    {
        add_action('do_parse_request', [$this, 'parseRequest'], 30, 2);
    }

    /**
     * Parse request
     * @param mixed $doParse 
     * @param mixed $wp 
     * @return void 
     */
    public function parseRequest($doParse, $wp)
    {
        $currentUrl = trim(esc_url_raw(add_query_arg([])), '/');
        $homePath = trim(parse_url(home_url(), PHP_URL_PATH), '/');
        if ($homePath && strpos($currentUrl, $homePath) === 0) {
            $currentUrl = trim(substr($currentUrl, strlen($homePath)), '/');
        }
        
        /** Check if this is a HeidiPay Webhook call */
        if (preg_match('/^heidipay\-webhooks\/([^\/]+)\/([^\/]+)/', $currentUrl, $matches) && is_array($matches) && count($matches) >= 3) {            
            $this->callHook($matches[1], $matches[2]);
            return false;
        }
        return $doParse;
    }

    /**
     * Generate URL
     * @param int $orderId 
     * @param string $hook 
     * @return string
     */
    public static function generateUrl(int $orderId, string $hook)
    {
        return get_site_url().'/heidipay-webhooks/'.urlencode($hook).'/'.$orderId;
    }

    /**
     * Call hook
     * @param string $hook 
     * @param string $orderId
     * @return never
     */
    protected function callHook($hook, $orderId)
    {
        header('Content-Type: application/json; charset=utf-8');
        $json = file_get_contents('php://input');
        if (! empty($json)) {
            $data = json_decode($json, true);
            if (empty($data) || ! $data) {
                http_response_code(400);
                echo json_encode([
                    'status' => 'ERROR',
                    'message' => 'Error while decoding JSON: '.json_last_error_msg()
                ]);
                die;
            }
        } else {
            $data = $_POST;
        }

        if (empty($data['token'])) {
            http_response_code(400);
            echo json_encode([
                'status' => 'ERROR',
                'message' => 'Empty token'
            ]);
            die;
        }

        $webhook = WebhookObject::getByToken($data['token'], $hook);
        if (empty($webhook) || $webhook->orderId != $orderId || $webhook->hook != $hook) {
            /** To prevent data leak we send the exact same error if the token is not found or if it does not match with order/hook */
            /** Thus, a hacker will not be able to know if the token really exists in our database or not. */
            http_response_code(404);
            echo json_encode([
                'status' => 'ERROR',
                'message' => 'Order not found with this token'
            ]);
            die;
        }

        $webhook->execute($data);

        http_response_code(500);
        echo json_encode([
            'status' => 'ERROR',
            'message' => 'Something went wrong...'
        ]);
        die;
    }
}

