<?php

/**
 * Copyright (c) HeidiPay.
 */

namespace HeidiPay\Classes;

class API {

    /** @var array STATUS */
    const STATUS = [
        'INITIALISED' => 'initialised',
        'ONGOING' => 'application_ongoing',
        'ABANDONED' => 'abandoned',
        'CLOSED' => 'closed',
        'APPROVED' => 'performing',
        'DELAYED' => 'delayed',
        'OVERDUE' => 'overdue',
        'COMPLETED' => 'completed'
    ];

    /** @var string AUTH_URL */
    const AUTH_URL = 'auth/generate/';

    /** @var string INIT_TRANS_URL */
    const INIT_TRANS_URL = 'api/checkout/v1/init/';

    /** @var string CONTRACT_STATUS_URL */
    const CONTRACT_STATUS_URL = 'api/checkout/v1/status/';

    /** @var string CONTRACT_REFUND_URL */
    const CONTRACT_REFUND_URL = 'api/checkout/v1/refund/';

    /** @var string CONTRACT_API_AVAILABLE */
    const CONTRACT_API_AVAILABLE = 'api/checkout/v1/available/';

    /**
     * Get heidi domain
     * @return string
     */
    public static function getHeidiDomain()
    {
        $domain = "https://origination.heidipay.com/";
        $gateways = WC()->payment_gateways()->payment_gateways();
        if (empty($gateways['heylight'])) {
            return $domain;
        }

        /** Get gateway settings */
        $gateway = $gateways['heylight'];
        $gateway->init_settings();
        $sandbox = $gateway->settings['sandbox'];
        if ($sandbox == 'yes') {
            $domain = "https://sbx-origination.heidipay.io/";
        }
        return $domain;
    }

    /**
     * Get auth token
     * @param bool $isPro
     * @return string|bool
     */
    public static function getAuthToken(bool $isPro = false)
    {
        $gateways = WC()->payment_gateways()->payment_gateways();
        if (empty($gateways['heylight'])) {
            Log::getInstance()->warning('The plugin is not installed or active.');
            return false;
        }

        /** Get gateway settings */
        $gateway = $gateways['heylight'];
        $gateway->init_settings();
        $key = ($isPro ? 'heidi_secret_key_pro' : 'heidi_secret_key');

        $merchantKey = $gateway->settings[$key];
        if (!$merchantKey) {
            Log::getInstance()->warning('There is no merchant key in configuration');
            return false;
        }
        $body = ['merchant_key' => $merchantKey];
        $url = self::getHeidiDomain() . self::AUTH_URL;
        $request_util = new Requester();
        $request_util->addHeader('X-Client-Module-Version', Helper::getVersion());
        $request_util->setUrl($url);
        $response = $request_util->post($body);

        $response = json_decode($response['contents'], true);

        if (
            empty($response['status'])
            || $response['status'] !== 'success'
            || empty($response['data']['token'])
        ) {
            Log::getInstance()->error(
                'Unable to get the credential token. Response Status: ' . $response['status'] . "\n" . 
                wc_print_r($response, true)
            );
            return false;
        }

        Log::getInstance()->debug('Authentication token successfully received');
        return $response['data']['token'];
    }

    /**
     * Request refund
     * @param string $externalUuid 
     * @param float $amount 
     * @param string $currency
     * @param bool $isPro
     * @param WC_Order $order
     * @return bool
     */
    public static function requestRefund($externalUuid, $amount, $currency, $isPro, $order)
    {
        $authToken = API::getAuthToken($isPro);
        if (!$authToken) {
            return false;
        }

        $url = API::getHeidiDomain() . API::CONTRACT_REFUND_URL;
        $request_util = new Requester();
        $request_util->addHeader('Authorization', 'Token ' . $authToken);
        $request_util->addHeader('X-Client-Module-Version', Helper::getVersion());
        $request_util->setUrl($url);
        $result = $request_util->post([
            'external_uuid' => $externalUuid,
            'amount' => number_format($amount, 2, '.', ''),
            'currency' => $currency
        ]);

        if ((int) $result['code'] !== 200) {
            Log::getInstance()->error(
                'Refund request failed. Error code: ' . $result['code'] . "\n" .
                wc_print_r($result, true), 
                $order
            );
            return false;
        }

        Log::getInstance()->debug('Refund successfully sent', $order);
        return true;
    }

    /**
     * Init transaction
     * @param array $requestData 
     * @param string $authToken
     * @param WC_Order $order
     * @return array
     */
    public static function initTransaction(array $requestData, $authToken, $order)
    {
        $url = API::getHeidiDomain() . API::INIT_TRANS_URL;
        $request_util = new Requester();
        $request_util->addHeader('Authorization', 'Token ' . $authToken);
        $request_util->addHeader('X-Client-Module-Version', Helper::getVersion());
        $request_util->setUrl($url);
        $response = $request_util->post($requestData);
        Log::getInstance()->debug(
            'Init transaction. Parameters are : ' . "\n" .
            wc_print_r($requestData, true),
            $order
        );
        if ($response['code'] != 201 || empty($response['contents'])) {
            Log::getInstance()->error(
                'Init transaction request failed. Error code: ' . $response['code'] . "\n" .
                wc_print_r($response, true),
                $order
            );
            return $response;
        }
        Log::getInstance()->debug(
            'Init transaction successfully sent. Response: ' . "\n" .
            wc_print_r($response, true),
            $order
        );
        return json_decode($response['contents'], true);
    }

    /**
     * Check order status
     * @param string $authToken
     * @param string $external_contract_uuid
     * @param WC_Order $order
     * @return bool
     */
    public static function checkOrderStatus($authToken, $external_contract_uuid, $order)
    {
        $url = API::getHeidiDomain() . API::CONTRACT_STATUS_URL . $external_contract_uuid . '/';
        $request_util = new Requester();
        $request_util->addHeader('Authorization', 'Token ' . $authToken);
        $request_util->addHeader('X-Client-Module-Version', Helper::getVersion());
        $request_util->setUrl($url);
        $response = $request_util->get();

        if ($response['code'] >= 300) {
            Log::getInstance()->error(
                'Order status request failed. Error code: ' . $response['code'] . "\n" .
                wc_print_r($response, true), 
                $order
            );
            return false; // Must be 2XX status
        }

        $content = json_decode($response['contents'], true);

        Log::getInstance()->debug('Order status successfully checked. Status: ' . $content['status'], $order);
        if (isset($content['status']) && strtolower($content['status']) === API::STATUS['APPROVED']) {
            return true;
        }

        return false;
    }
}
