<?php

/**
 * Copyright (c) HeidiPay.
 */

namespace HeidiPay\Classes;

use HeidiPay\Controllers\Gateway;
use HeidiPay\Controllers\Webhook as ControllersWebhook;
use WC_Geolocation;
use WC_Integration_MaxMind_Geolocation;

class Helper {

    /**
     * Get version
     * @return string 
     */
    public static function getVersion()
    {
        return '2.7.1';
    }

    /**
     * Get init data
     * @param WC_Order $order
     * @param string $token
     * @param WC_Payment_Gateway $wc_payment_gateway
     */
    public static function getInitData($order, $token, $wc_payment_gateway)
    {
        /** Get global terms */
        $gateways = WC()->payment_gateways()->payment_gateways();
        $gateway = null;
        $globalTerms = Widget::DEFAULT_WIDGET_VALUES['widget_terms'];
        if (! empty($gateways['heylight'])) {
            /** Get gateway settings */
            $gateway = $gateways['heylight'];
            $gateway->init_settings();
            $globalTerms = (! empty($gateway->settings['widget_terms']) ? (is_array($gateway->settings['widget_terms']) ? $gateway->settings['widget_terms'] : json_decode($gateway->settings['widget_terms'])) : Widget::DEFAULT_WIDGET_VALUES['widget_terms']);
        }

        /** Check BNPL Product */
        $filter = Filter::getOrderAllowedTerms($order, true);
        $code = null;
        if (! empty($filter)) {
            $bnplProductId = $filter[Filter::META_PREFIX.'_product'];
            foreach (Filter::PRODUCT_TYPES as $productType) {
                if ($productType['id'] == $bnplProductId) {
                    $bnplProduct = $productType;
                    break;
                }
            }
            $code = $filter[Filter::META_PREFIX.'_code'];
        } else {
            $bnplProduct = Widget::getGeneralBNPLProduct();
            if (empty($bnplProduct)) {
                $types = Filter::PRODUCT_TYPES;
                $bnplProduct = reset($types);
            }
        }
        $isPro = (bool) $bnplProduct['pro'];
        if ($isPro && ! Widget::isProAvailable()) {
            $isPro = false;
        }
        if ($isPro && empty($code)) {
            $code = (! empty($gateway->settings['CODE']) ? $gateway->settings['CODE'] : '-');
        }

        /** Get order specific terms */
        $orderTerms = $globalTerms;
        $filteredTerms = Filter::getOrderAllowedTerms($order);
        if ($filteredTerms !== null) {
            $orderTerms = $filteredTerms;
        }

        /** Check amount */
        $total = (float) $order->get_total();
        $amount = number_format($total, 2, '.', '');
        if (! Helper::checkAmount($amount, $isPro, $orderTerms)) {
            $isPro = !$isPro;
            if (! Helper::checkAmount($amount, $isPro, $orderTerms)) {
                return false;
            }
        }
        if ($isPro && ! Widget::isProAvailable()) {
            return false;
        }

        /** Recheck if terms are allowed */
        if ($isPro) {
            $minimumInstalmentPrice = (!empty($gateway->settings['widget_min_instalment_pro']) ? (float) $gateway->settings['widget_min_instalment_pro'] : Widget::DEFAULT_WIDGET_VALUES['widget_min_instalment_pro']);
        } else {
            $minimumInstalmentPrice = (!empty($gateway->settings['widget_min_instalment']) ? (float) $gateway->settings['widget_min_instalment'] : Widget::DEFAULT_WIDGET_VALUES['widget_min_instalment']);
        }
        $allowedTerms = Helper::getAllowedTerms($orderTerms, (int) round($amount * 100, 0), $minimumInstalmentPrice);
        if (empty($allowedTerms)) {
            return false;
        }

        /** Format array to get an integer array without keys */
        $allowedTerms = array_values(array_map('intval', $allowedTerms));
        
        $reference = (int) $order->get_id();
        if (! empty($gateway->settings['order_ref_sent']) && $gateway->settings['order_ref_sent'] === 'reference') {
            $reference = $order->get_order_number();
        }

        $requestData = [
            'amount_format'       => 'DECIMAL',
            'allowed_terms'       => $allowedTerms,
            'amount'              => [
                'amount'          => self::getFormattedPrice($order->get_total()),
                'currency'        => get_woocommerce_currency(),
            ],
            'redirect_urls'       => [
                'success_url'     => self::getWebhookUrl($order->get_id()),
                'failure_url'     => self::getWebhookUrl($order->get_id(), false),
            ],
            'webhooks' => [
                'status_url' => ControllersWebhook::generateUrl((int) $order->get_id(), Webhook::HOOK_ORDER_STATUS),
                'mapping_scheme' => "WOOCOMMERCE",
                'token' => Webhook::getTokenForOrder((int) $order->get_id(), Webhook::HOOK_ORDER_STATUS)
            ],
            'origination_channel' => 'ecommerce',
            'order_reference'     => $reference,
            'customer_details'    => [
                'email_address'   => $order->get_billing_email(),
                'first_name'      => $order->get_billing_first_name(),
                'last_name'       => $order->get_billing_last_name(),
            ],
            // Shipping address is empty by default
            // 'shipping_address' => [
            // 'zip_code' => $order->get_shipping_postcode(),
            // 'city' => $order->get_shipping_city(),
            // 'address_line_1' => $order->get_shipping_address_1(),
            // 'country_code' => HEIDIPAY_ISO3_COUNTRIES[ $order->get_shipping_country() ],
            // ],
            'products'            => self::getProducts($order, $wc_payment_gateway),
            // 'shipping_carrier' => $order->get_shipping_method(),
            'financial_product_type' => ($isPro ? 'CREDIT' : 'BNPL')
        ];

        if (false) {
            $acceptedLanguages = ['en', 'fr', 'de', 'it'];
            $language = self::tryToGetLanguageIso($order);
            if (! empty($language) && in_array($language, $acceptedLanguages)) {
                $requestData['language'] = $language;
            }
        }

        $address = $order->get_billing_address_1();
        if (! empty($address)) {
            $requestData['billing_address'] = [
                'zip_code'        => $order->get_billing_postcode(),
                'city'            => $order->get_billing_city(),
                'address_line_1'  => $order->get_billing_address_1(),
                'country_code'    => 'ITA',
            ];
        }

        /** set $code if PRO */
        if ($isPro && ! empty($code)) {
            $requestData['additional_data'] = ['pricing_structure_code' => $code];
        }

        if ($order->get_billing_address_2()) {
            $requestData['billing_address']['address_line_2'] = $order->get_billing_address_2();
        }

        // if ($order->get_shipping_address_2()) {
        // $requestData['shipping_address']['address_line_2'] = $order->get_shipping_address_2();
        // }

        return $requestData;
    }

    /**
     * Try to get language ISO
     * @param WC_Order $order
     * @return string|null $languageIso
     */
    protected static function tryToGetLanguageIso($order)
    {
        /** WPML from Order object */
        $language = get_post_meta($order->get_id(), 'wpml_language', true);
        if (! empty($language)) {
            return $language;
        }

        /** WPML Current language */
        $language = apply_filters('wpml_current_language', null);
        if (! empty($language)) {
            return $language;
        }

        /** WeGlot current language */
        if (function_exists('weglot_get_current_language')) {
            $language = weglot_get_current_language();
            if (! empty($language)) {
                return $language;
            }
        }

        /** Polylang current language */
        if (function_exists('pll_current_language')) {
            $language = pll_current_language();
            if (! empty($language)) {
                return $language;
            }
        }

        /** Not found */
        return null;
    }

    /**
     * Get products
     * @param WC_Order $order
     * @param WC_Payment_Gateway $wc_payment_gateway
     * @return array
     */
    protected static function getProducts($order, $wc_payment_gateway)
    {
        global $product;
        $items = [];

        /** Get all items */
        foreach ($order->get_items() as $item) {
            if ($item['variation_id']) {
                $product = wc_get_product($item->get_variation_id());
            } else {
                $product = wc_get_product($item->get_product_id());
            }

            $quantity = (int) (! empty($item->get_quantity()) ? $item->get_quantity() : 1);
            $unitPrice = ($item->get_total() + $item->get_total_tax()) / $quantity;
            $tmp = [
                'sku'      => ($product && ! empty($product) ? $product->get_sku() : ''),
                'quantity' => $quantity,
                'name'     => $item->get_name(),
                'price'    => self::getFormattedPrice($unitPrice)
            ];

            $items[] = $tmp;
        }

        return $items;
    }

    /**
     * Get webhook url
     * @param mixed $orderId 
     * @param bool $success 
     * @return string 
     */
    public static function getWebhookUrl($orderId, $success = true)
    {
        $status = $success ? 'heidipay_success' : 'heidipay_fail';
        $statusUrl = WC()->api_request_url($status);
        return add_query_arg('id', $orderId, $statusUrl);
    }

    /**
     * Convert price to a valid format 123456.78
     *
     * @param string|float|int $price
     *
     * @return string
     */
    public static function getFormattedPrice($price)
    {
        return number_format($price, 2, '.', '');
    }

    /**
     * Get allowed term
     * @param array $terms 
     * @param int $priceDisplayed 
     * @param int $minimumInstalmentPrice 
     * @return array
     */
    public static function getAllowedTerms($terms, $priceDisplayed, $minimumInstalmentPrice)
    {
        $maximumTerms = floor(($priceDisplayed / 100) / $minimumInstalmentPrice);
        $terms = array_map('intval', $terms);
        sort($terms, SORT_NUMERIC);
        $allowedTerms = [];
        foreach ($terms as $term) {
            if ($term <= $maximumTerms) {
                $allowedTerms[] = $term;
            }
        }
        return $allowedTerms;
    }

    /**
	 * Add fee
	 * @param float $minorAmount 
	 * @return float|false
	 */
	public static function addFee($minorAmount)
	{
        $gateways = WC()->payment_gateways()->payment_gateways();
        if (empty($gateways['heylight'])) {
            return false;
        }

        /** Get gateway settings */
        $gateway = $gateways['heylight'];
        $gateway->init_settings();
		$fee = (! empty($gateway->settings['widget_fee']) ? (float) str_replace(',', '.', $gateway->settings['widget_fee']) : 0);
        if ($fee > 0) {
            $minorAmount = round($minorAmount + ($minorAmount * ($fee / 100)));
        }
		return $minorAmount;
	}

    /**
     * Check amount
     * @param float $amount 
     * @param bool $isPro
     * @param array $terms
     * @return bool 
     */
    public static function checkAmount($amount, bool $isPro, $terms = null)
    {
        if ($amount <= 0) {
            return false;
        }

        $gateways = WC()->payment_gateways()->payment_gateways();
        if (empty($gateways['heylight'])) {
            // echo "<script>console.debug('Price-per-month: Could not find gateway' );</script>";
            return false;
        }

        /** Get gateway settings */
        $gateway = $gateways['heylight'];
        $gateway->init_settings();

        /** Add fee */
        $fee =  (!empty($gateway->settings['widget_fee']) ? $gateway->settings['widget_fee'] : Widget::DEFAULT_WIDGET_VALUES['widget_fee']);
        if ($fee > 0) {
            $amount = round($amount + ($amount * ($fee / 100)));
        }

        /** Check minimum price */
        if ($isPro) {
            $minPrice = (!empty($gateway->settings['widget_min_amount_pro']) ? $gateway->settings['widget_min_amount_pro'] : Widget::DEFAULT_WIDGET_VALUES['widget_min_amount_pro']);
            $maxPrice = (!empty($gateway->settings['widget_max_amount_pro']) ? $gateway->settings['widget_max_amount_pro'] : Widget::DEFAULT_WIDGET_VALUES['widget_max_amount_pro']);
            $minimumInstalmentPrice = (int) (empty($gateway->settings['widget_min_instalment_pro']) ? Widget::DEFAULT_WIDGET_VALUES['widget_min_instalment_pro'] : $gateway->settings['widget_min_instalment_pro']);
        } else {
            $minPrice = (!empty($gateway->settings['widget_min_amount']) ? $gateway->settings['widget_min_amount'] : Widget::DEFAULT_WIDGET_VALUES['widget_min_amount']);
            $maxPrice = (!empty($gateway->settings['widget_max_amount']) ? $gateway->settings['widget_max_amount'] : Widget::DEFAULT_WIDGET_VALUES['widget_max_amount']);
            $minimumInstalmentPrice = (int) (empty($gateway->settings['widget_min_instalment']) ? Widget::DEFAULT_WIDGET_VALUES['widget_min_instalment'] : $gateway->settings['widget_min_instalment']);
        }
        
        if ($minPrice > $amount || ($maxPrice > 0 && $maxPrice < $amount)) {
            return false;
        }

        /** Check terms */
        $terms = (empty($gateway->settings['widget_terms']) ? Widget::DEFAULT_WIDGET_VALUES['widget_terms'] : (is_array($gateway->settings['widget_terms']) ? $gateway->settings['widget_terms'] : json_decode($gateway->settings['widget_terms'])));
        $allowedTerms = self::getAllowedTerms($terms, round($amount * 100), $minimumInstalmentPrice);
        if (empty($allowedTerms)) {
            return false;
        }

        return true;
    }

    /**
     * Is MaxMind Set
     * @return bool
     */
    public static function isMaxMindSet()
    {
        $maxMind = new WC_Integration_MaxMind_Geolocation();
        $key = $maxMind->get_option('license_key');
        return ! empty($key);
    }

    /**
     * Is country restricted
     * @return bool
     */
    public static function isCountryRestricted()
    {
        $gateways = WC()->payment_gateways()->payment_gateways();
        if (empty($gateways['heylight'])) {
            return true;
        }

        /** Check MaxMind API Key is set */
        if (! self::isMaxMindSet()) {
            return false; 
        }

        /** Get gateway settings */
        $gateway = $gateways['heylight'];
        $gateway->init_settings();
        if (empty($gateway->settings['countries'])) {
            /** Do not restrict by default */
            return false;
        }

        /** Get country linked to customer billing address */
        if (! empty(WC()->customer)) {
            $country = WC()->customer->get_billing_country();
            if (! empty($country)) {
                /** Check if country is restricted */
                if (! in_array($country, array_keys(Gateway::COUNTRY_LIST))) {
                    return false;
                }
                return ! in_array($country, $gateway->settings['countries']);
            }
        }

        /** Get geolocation */
        $location = WC_Geolocation::geolocate_ip();
        if (empty($location['country'])) {
            /** Do not restrict by default */
            return false;
        }

        /** Check if country is restricted */
        if (! in_array($location['country'], array_keys(Gateway::COUNTRY_LIST))) {
            return false;
        }
        return ! in_array($location['country'], $gateway->settings['countries']);
    }
}
