<?php

/**
 * Copyright (c) HeidiPay.
 */

namespace HeidiPay\Classes;

class Log {

    /** @var Log $instance */
    protected static $instance;
    
    /** @var array CONTEXT */
    const CONTEXT = ['source' => 'heylight'];

    /** @var WC_Logger_Interface $logger */
    protected $logger;

    /**
     * Get instance
     * @return Log 
     */
    public static function getInstance()
    {
        if (empty(self::$instance)) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor.
     * @return void 
     */
    public function __construct()
    {
        $this->logger = wc_get_logger();
    }

    /**
     * Format message
     * @param string $message
     * @param WC_Order|null $order
     * @return string $message
     */
    protected function formatMessage(string $message, $order = null)
    {
        if (! empty($order) 
            && is_callable([$order, 'add_order_note']) 
            && is_callable([$order, 'get_id'])
        ) {
            $message = '[ORDER #'.$order->get_id().'] '.$message;
            $order->add_order_note($message);
        }
        return $message;
    }

    /**
     * Info
     * @param string $message 
     * @param WC_Order|null $order
     * @return void 
     */
    public function info(string $message, $order = null)
    {
        $message = $this->formatMessage($message, $order);
        $this->logger->info($message, self::CONTEXT);
    }

    /**
     * Emergency
     * @param string $message 
     * @param WC_Order|null $order
     * @return void 
     */
    public function emergency(string $message, $order = null)
    {
        $message = $this->formatMessage($message, $order);
        $this->logger->emergency($message, self::CONTEXT);
    }

    /**
     * Alert
     * @param string $message 
     * @param WC_Order|null $order
     * @return void 
     */
    public function alert(string $message, $order = null)
    {
        $message = $this->formatMessage($message, $order);
        $this->logger->alert($message, self::CONTEXT);
    }

    /**
     * Critical
     * @param string $message 
     * @param WC_Order|null $order
     * @return void 
     */
    public function critical(string $message, $order = null)
    {
        $message = $this->formatMessage($message, $order);
        $this->logger->critical($message, self::CONTEXT);
    }

    /**
     * Error
     * @param string $message 
     * @param WC_Order|null $order
     * @return void 
     */
    public function error(string $message, $order = null)
    {
        $message = $this->formatMessage($message, $order);
        $this->logger->error($message, self::CONTEXT);
    }

    /**
     * Warning
     * @param string $message 
     * @param WC_Order|null $order
     * @return void 
     */
    public function warning(string $message, $order = null)
    {
        $message = $this->formatMessage($message, $order);
        $this->logger->warning($message, self::CONTEXT);
    }

    /**
     * Notice
     * @param string $message 
     * @param WC_Order|null $order
     * @return void 
     */
    public function notice(string $message, $order = null)
    {
        $message = $this->formatMessage($message, $order);
        $this->logger->notice($message, self::CONTEXT);
    }

    /**
     * Debug
     * @param string $message 
     * @param WC_Order|null $order
     * @return void 
     */
    public function debug(string $message, $order = null)
    {
        $message = $this->formatMessage($message, $order);
        $this->logger->debug($message, self::CONTEXT);
    }

}