<?php

/**
 * Copyright (c) HeidiPay.
 */

namespace HeidiPay\Classes;

class Widget {

    /** @var Widget */
	private static $instance;

    /** @var string LIBRARY_URL */
    const LIBRARY_URL = '/sdk/heidi-upstream-lib.js';

    /** @var string CADENCE */
    const CADENCE = 'MONTHLY';

    /** @var bool IS_RESTRICTED */
    const IS_RESTRICTED = true;

    /** @var bool CAN_BE_PRO */
    const CAN_BE_PRO = true;

    /** @var bool CAN_DISABLE_QR */
    const CAN_DISABLE_QR = false;

    /** @var bool PRO_NEED_CODE */
    const PRO_NEED_CODE = true;

    /** @var bool PPM_ONLY */
    const PPM_ONLY = false;

    /** @var array PRO_TERMS */
    const PRO_TERMS = [25, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36];

    /** @var array AVAILABLE_TERMS */
    const AVAILABLE_TERMS = [3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24];

    /** @var string RESTRICTED_CURRENCY */
    const RESTRICTED_CURRENCY = '';

    /** @var array RESTRICTED_VALUES */
    const RESTRICTED_VALUES = [
        'widget_fee',
        // 'widget_min_amount',
        // 'widget_min_instalment',
    ];

    /** @var array DEFAULT_WIDGET_VALUES */
    const DEFAULT_WIDGET_VALUES = [
        'widget_color' => 'red',
        'widget_enabled' => true,
        'widget_enabled_cart' => true,
        'widget_terms' => ["6", "12"],
        'widget_fee' => 0,
        'widget_min_amount' => 60,
        'widget_max_amount' => 3000,
        'widget_min_instalment' => 20,
        'widget_min_amount_pro' => 30,
        'widget_max_amount_pro' => 5000,
        'widget_min_instalment_pro' => 20,
        'widget_with_card' => true,
        'PRODUCT' => 'HEYLIGHT_BNPL_IT',
        'CODE' => ''
    ];

    /**
     * Constructor.
     * @return void
     */
    public function __construct()
    {
        add_filter('woocommerce_get_price_html', [$this, 'loadProductWidget'], 10, 2);
        add_action('woocommerce_cart_totals_after_order_total', [$this, 'loadCartWidget']);
        add_action('woocommerce_review_order_after_order_total', [$this, 'loadCartWidget']);
        add_action('wp_enqueue_scripts', [$this, 'registerAssets']);
        add_shortcode('heylight_widget', [$this, 'loadShortcodeWidget']);
    }

    /**
     * Register assets
     * @return void
     */
    public function registerAssets()
    {
        wp_register_script(
            'heidipay_widget-library',
            self::getHeidiDomain().self::LIBRARY_URL,
            ['jquery']
        );
    }

    /**
     * Load cart wiget
     * @return string
     */
    public function loadCartWidget()
    {
        $cartTotal = WC()->cart->get_cart_contents_total() + WC()->cart->get_shipping_total();
        if (! is_numeric($cartTotal)) {
            $cartTotal = 0;
        }
        if (get_option('woocommerce_tax_display_cart') == 'incl') {
            $cartTotal += WC()->cart->get_cart_contents_tax() + WC()->cart->get_shipping_tax();
        }
        if (! is_numeric($cartTotal)) {
            $cartTotal = 0;
        }
        $html = $this->getHtml([
            'minorAmount' => round(floatval($cartTotal) * 100),
            'reloadLibrary' => true,
            'object' => WC()->cart
        ], true);
        echo (empty($html) ? '' : '<tr><td colspan="2">' . $html . '</td></tr>');
    }

    /**
     * Load product widget
     * @param string $html
     * @param object $product
     * @return string
     */
    public function loadProductWidget($html, $product)
    {
        if (
            $product->is_type('variable')
            || is_a($product, 'WC_Product_Variation')
        ) {
            if (!is_a($product, 'WC_Product_Variation')) {
                $allVariations = $product->get_available_variations();
                $hasSamePrices = true;
                foreach ($allVariations as $productVariation) {
                    if (!empty($productVariation['price_html'])) {
                        $hasSamePrices = false;
                        break;
                    }
                }
                if ($hasSamePrices && get_queried_object_id() == $product->get_id()) {
                    $html .= $this->getHtml([
                        'minorAmount' => (int) $this->getProductPrice($product),
                        'reloadLibrary' => false,
                        'object' => $product
                    ]);
                    wp_enqueue_script('heidipay_widget-library');
                }
                return $html;
            }
            return $this->loadVariableWidget($html, $product);
        }

        if (get_queried_object_id() != $product->get_id()) {
            return $html;
        }

        if (is_a($product, 'WC_Product_Bundle')) {
            $html .= $this->getHtml([
                'minorAmount' => (int) $this->getProductBundlePrice($product),
                'reloadLibrary' => true,
                'object' => $product
            ]);
            return $html;
        }

        $html .= $this->getHtml([
            'minorAmount' => (int) $this->getProductPrice($product),
            'reloadLibrary' => true,
            'object' => $product
        ]);
        return $html;
    }

    /**
     * Load variable widget
     * @param string $html
     * @param object $product
     * @return string
     */
    public function loadVariableWidget($html, $product)
    {
        if (get_queried_object_id() != $product->get_parent_id()) {
            return $html;
        }

        $html .= $this->getHtml([
            'minorAmount' => (int) $this->getProductPrice($product),
            'reloadLibrary' => true,
            'object' => $product
        ]);
        return $html;
    }

    /**
     * Load shortcode widget
     * @param array $attributes
     * @return string $html
     */
    public function loadShortcodeWidget($attributes)
    {
        /**
         * Usage example: 
         * [heylight_widget amount="123.45"]
         */
        $attributes = shortcode_atts( [
            'amount' => null
        ], $attributes );
        return $this->getHtml([
            'minorAmount' => (int) round((float) $attributes['amount'] * 100),
            'reloadLibrary' => true,
            'object' => null
        ]);
    }

    /**
     * Get HTML for parameters
     * @param array $parameters
     * @param bool $forCart
     * @return string
     */
    private function getHtml($parameters, $forCart = false)
    {
        if ((int) $parameters['minorAmount'] < 0) {
            return;
        }

        $configuration = $this->getConfigurationParameters((int) $parameters['minorAmount'], $forCart, $parameters['object']);
        if (!$configuration) {
            return;
        }

        if (! empty(self::RESTRICTED_CURRENCY) && get_woocommerce_currency() !== self::RESTRICTED_CURRENCY) {
            return;
        }

        $parameters = array_merge(
            $parameters,
            $this->getCurrencyParameters(),
            $this->getLanguageParameters(),
            $configuration
        );

        if ((!$forCart && $parameters['enabled'] !== 'yes') || ($forCart && $parameters['enabled_cart'] !== 'yes')) {
            return;
        }

        $html = '<div id="heidipay-container" class="heidipay-container-2"
            data-heidipay="true"
            data-heidipay-minorAmount="' . $parameters['minorAmount'] . '"
            data-heidipay-currencySymbol="' . $parameters['currencySymbol'] . '"
            data-heidipay-lang="' . $parameters['lang'] . '"
            data-heidipay-type="' . $parameters['type'] . '"
            data-heylight-logo-variant="' . $parameters['color'] . '"
            data-heidipay-apiKey="' . $parameters['apiKey'] . '"
            data-heidipay-cadence="' . self::CADENCE . '"
            data-heidipay-thousandsSeparator="' . $parameters['thousandsSeparator'] . '"
            data-heidipay-decimalSeparator="' . $parameters['decimalSeparator'] . '"
            data-heidipay-symbolOnLeft="' . $parameters['symbolOnLeft'] . '"
            data-heidipay-spaceBetweenAmountAndSymbol="' . $parameters['spaceBetweenAmountAndSymbol'] . '"
            data-heidipay-decimalDigits="' . $parameters['decimalDigits'] . '"';
            if ($parameters['isPro']) {
                $html .= '
                    data-heidipay-allowedTerms="'.implode(',', $parameters['allowedTerms']).'"
                ';
                if (self::PRO_NEED_CODE) {
                    $html .= '
                        data-heidipay-pricing-structure-code="'.$parameters['code'].'"
                    ';
                }
            } else {
                $html .= '
                    data-heidipay-term="'.$parameters['bestTerm'].'"
                ';
            }
        $html .= '></div>';

        if (
            !empty($parameters['reloadLibrary'])
            && $parameters['reloadLibrary'] == true
        ) {
            $html .= '<script type="text/javascript" src="' . self::getHeidiDomain().self::LIBRARY_URL . '" defer></script>';
        }

        if ($forCart) {
            $html .= '<script type="text/javascript" defer>
                (function ($) {
                    $(function () {
                        $(document).on("updated_cart_totals", function () {
                            var script = $("<script>");
                            script.attr("src", "' . self::getHeidiDomain().self::LIBRARY_URL . '");
                            $("head").append(script);
                        });
                    });
                })(jQuery);
            </script>';
        }

        return $html;
    }

    /**
     * Get currency parameters
     * @return array
     */
    private function getCurrencyParameters()
    {
        $parameters = [];
        $parameters = apply_filters(
            'wc_price_args',
            wp_parse_args(
                $parameters,
                array(
                    'ex_tax_label'       => false,
                    'currency'           => '',
                    'decimal_separator'  => wc_get_price_decimal_separator(),
                    'thousand_separator' => wc_get_price_thousand_separator(),
                    'decimals'           => wc_get_price_decimals(),
                    'price_format'       => get_woocommerce_price_format(),
                )
            )
        );

        $parameters['currencySymbol'] = get_woocommerce_currency_symbol($parameters['currency']);
        $parameters['thousandsSeparator'] = $parameters['thousand_separator'];
        $parameters['decimalSeparator'] = $parameters['decimal_separator'];
        $parameters['decimalDigits'] = (int) $parameters['decimals'];

        $currencyPosition = get_option('woocommerce_currency_pos');
        $parameters['symbolOnLeft'] = 'true';
        $parameters['spaceBetweenAmountAndSymbol'] = 'false';
        switch ($currencyPosition) {
            case 'left':
                $parameters['symbolOnLeft'] = 'true';
                $parameters['spaceBetweenAmountAndSymbol'] = 'false';
                break;
            case 'right':
                $parameters['symbolOnLeft'] = 'false';
                $parameters['spaceBetweenAmountAndSymbol'] = 'false';
                break;
            case 'left_space':
                $parameters['symbolOnLeft'] = 'true';
                $parameters['spaceBetweenAmountAndSymbol'] = 'true';
                break;
            case 'right_space':
                $parameters['symbolOnLeft'] = 'false';
                $parameters['spaceBetweenAmountAndSymbol'] = 'true';
                break;
        }

        return $parameters;
    }

    /**
     * Get language parameters
     * @return array
     */
    private function getLanguageParameters()
    {
        return [
            'lang' => substr(get_bloginfo('language'), 0, 2)
        ];
    }

    /**
     * Get configuration parameters
     * @param int $minorAmount
     * @param bool $forCart
     * @param object|null $object
     * @return array|false configuration parameters
     */
    private function getConfigurationParameters($minorAmount, $forCart, $object = null)
    {
        $gateways = WC()->payment_gateways()->payment_gateways();
        if (empty($gateways['heylight'])) {
            return false;
        }

        /** Get gateway settings */
        $gateway = $gateways['heylight'];
        $gateway->init_settings();
        if ($gateway->enabled != 'yes') {
            return false;
        }

        /** Check country */
        if (Helper::isCountryRestricted()) {
            return false;
        }

        $terms = null;
        $filter = null;
        if ($object !== null) {
            /** Check product/category Filter (Disabled) */
            if ($forCart) {
                if (! Filter::shouldCartDisplayWidget($object)) {
                    return false;
                }
            } else {
                if (! Filter::shouldProductDisplayWidget($object)) {
                    return false;
                }
            }

            /** Check product/category Filter (Term) */
            if ($forCart) {
                $terms = Filter::getCartAllowedTerms($object);
                $filter = Filter::getCartAllowedTerms($object, true);
            } else {
                $terms = Filter::getProductAllowedTerms($object);
                $filter = Filter::getProductAllowedTerms($object, true);
            }
        }

        if ($terms === null) {
            $terms = (!empty($gateway->settings['widget_terms']) ? (is_array($gateway->settings['widget_terms']) ? $gateway->settings['widget_terms'] : json_decode($gateway->settings['widget_terms'])) : self::DEFAULT_WIDGET_VALUES['widget_terms']);
        }

        /** Check if PRO or not */
        $bnplProduct = null;
        $isPro = null;
        if (! empty($filter) && ! empty($filter[Filter::META_PREFIX.'_product'])) {
            $bnplProductId = $filter[Filter::META_PREFIX.'_product'];
        } else {
            $bnplProductId = self::getGeneralBNPLProduct()['id'];
        }
        foreach (Filter::PRODUCT_TYPES as $productType) {
            if ($productType['id'] == $bnplProductId) {
                $bnplProduct = $productType;
                break;
            }
        }
        if (! empty($bnplProduct)) {
            $isPro = (bool) $bnplProduct['pro'];
        } else {
            $isPro = false;
        }
        
        if ($isPro) {
            $minimumInstalmentPrice = (!empty($gateway->settings['widget_min_instalment_pro']) ? (float) $gateway->settings['widget_min_instalment_pro'] : self::DEFAULT_WIDGET_VALUES['widget_min_instalment_pro']);
        } else {
            $minimumInstalmentPrice = (!empty($gateway->settings['widget_min_instalment']) ? (float) $gateway->settings['widget_min_instalment'] : self::DEFAULT_WIDGET_VALUES['widget_min_instalment']);
        }
        $allowedTerms = Helper::getAllowedTerms($terms, $minorAmount, $minimumInstalmentPrice);
        if (empty($allowedTerms)) {
            return false;
        }

        /** Check available */
        if ($isPro && ! self::isProAvailable()) {
            $isPro = false;
            $code = null;
        } else if (! $isPro && ! self::isRegularAvailable()) {
            $isPro = true;
        }

        /** Check amount */
        if (! Helper::checkAmount($minorAmount / 100, $isPro, $terms)) {
            $isPro = !$isPro;
            $filter = null;
            if (! Helper::checkAmount($minorAmount / 100, $isPro, $terms)) {
                return false;
            }
        }
        $code = null;
        if ($isPro && $filter === null) {
            $code = (! empty($gateway->settings['CODE']) ? $gateway->settings['CODE'] : '-');
        } else if ($isPro) {
            $code = $filter[Filter::META_PREFIX.'_code'];
            if (empty($code)) {
                $code = (! empty($gateway->settings['CODE']) ? $gateway->settings['CODE'] : '-');
            }
        }

        /** Check available */
        if ($isPro && ! self::isProAvailable()) {
            return false;
        } else if (! $isPro && ! self::isRegularAvailable()) {
            return false;
        }

        /** Reset BNPL Product */
        if ($bnplProduct['pro'] != $isPro) {
            foreach (Filter::PRODUCT_TYPES as $productType) {
                if ($productType['pro'] == $isPro) {
                    $bnplProduct = $productType;
                    break;
                }
            }
        }

        /** Calculate minorAmount with fee */
        $minorAmount = Helper::addFee($minorAmount);

        /** Get widget type */
        $type = "PRODUCT_DESCRIPTION_";
        $type .= $bnplProduct['id'];
        if ($type === 'PRODUCT_DESCRIPTION_HEIDIPAY' || $type === 'PRODUCT_DESCRIPTION_' ) {
            $type = 'PRODUCT_DESCRIPTION';
        }
        $fee = (! empty($gateway->settings['widget_fee']) ? (float) str_replace(',', '.', $gateway->settings['widget_fee']) : 0);
        if ($fee > 0) {
            $type .= '_WITH_CHARGE';
        }

        $withCard = (! empty($gateway->settings['widget_with_card']) ? $gateway->settings['widget_with_card'] : self::DEFAULT_WIDGET_VALUES['widget_with_card']);
        if (self::CAN_DISABLE_QR && $withCard !== true && $withCard !== 'yes' && ! $isPro) {
            $type .= '_CARD';
        }

        /** Send parameters */
        $parameters = [
            'color' => (!empty($gateway->settings['widget_color']) ? $gateway->settings['widget_color'] : self::DEFAULT_WIDGET_VALUES['widget_color']),
            'enabled' => (!empty($gateway->settings['widget_enabled']) ? $gateway->settings['widget_enabled'] : false),
            'enabled_cart' => (!empty($gateway->settings['widget_enabled_cart']) ? $gateway->settings['widget_enabled_cart'] : false),
            'bestTerm' => end($allowedTerms),
            'allowedTerms' => $allowedTerms,
            'isPro' => $isPro,
            'code' => $code,
            'minorAmount' => $minorAmount,
            'type' => $type,
            'apiKey' => (!empty($gateway->settings['widget_api_key']) ? $gateway->settings['widget_api_key'] : '-')
        ];
        return $parameters;
    }

    /**
     * Get general BNPL product
     * @return array|null
     */
    public static function getGeneralBNPLProduct()
    {
        $gateways = WC()->payment_gateways()->payment_gateways();
        if (empty($gateways['heylight'])) {
            return false;
        }

        /** Get gateway settings */
        $gateway = $gateways['heylight'];
        $gateway->init_settings();

        $generalProduct = (! empty($gateway->settings['PRODUCT']) ? $gateway->settings['PRODUCT'] : '-');
        $firstProduct = null;
        foreach (Filter::PRODUCT_TYPES as $product) {
            if (empty($firstProduct)) {
                $firstProduct = $product;
            }
            if ($product['id'] === $generalProduct) {
                return $product;
            }
        }
        return $firstProduct;
    }

    /**
     * Is PRO available ?
     * @return bool
     */
    public static function isProAvailable()
    {
        if (self::PPM_ONLY) {
            return true;
        }
        if (! self::CAN_BE_PRO) {
            return false;
        }
        $gateways = WC()->payment_gateways()->payment_gateways();
        if (empty($gateways['heylight'])) {
            return false;
        }

        /** Get gateway settings */
        $gateway = $gateways['heylight'];
        $gateway->init_settings();

        $merchantKey = $gateway->settings['heidi_secret_key_pro'];
        if (empty($merchantKey)) {
            return false;
        }

        return true;
    }

    /**
     * Is Regular available ?
     * @return bool
     */
    public static function isRegularAvailable()
    {
        if (self::PPM_ONLY) {
            return true;
        }
        $gateways = WC()->payment_gateways()->payment_gateways();
        if (empty($gateways['heylight'])) {
            return false;
        }

        /** Get gateway settings */
        $gateway = $gateways['heylight'];
        $gateway->init_settings();

        $merchantKey = $gateway->settings['heidi_secret_key'];
        if (empty($merchantKey)) {
            return false;
        }

        return true;
    }

    /**
     * Get heidi domain
     * @return string
     */
    public static function getHeidiDomain()
    {
        $domain = "https://upstream.heidipay.com";
        $gateways = WC()->payment_gateways()->payment_gateways();
        if (empty($gateways['heylight'])) {
            return false;
        }

        /** Get gateway settings */
        $gateway = $gateways['heylight'];
        $gateway->init_settings();
        $sandbox = $gateway->settings['sandbox'];
        if ($sandbox == 'yes') {
            $domain = "https://sbx-upstream.heidipay.io";
        }
        return $domain;
    }

    /**
     * Get product price
     * @param object $product
     * @return int $price * 100
     */
    private function getProductPrice($product)
    {
        if (get_option('woocommerce_tax_display_shop') == 'incl') {
            $price = wc_get_price_including_tax($product);
        } else {
            $price = wc_get_price_excluding_tax($product);
        }
        if (!is_numeric($price)) {
            $price = 0;
        }
        return (int) round($price * 100);
    }

    /**
     * Get product bundle price
     * @param object $product
     * @return int $price * 100
     */
    private function getProductBundlePrice($product)
    {
        if (get_option('woocommerce_tax_display_shop') == 'incl') {
            $price = $product->get_bundle_price_including_tax();
        } else {
            $price = $product->get_bundle_price_excluding_tax();
        }
        if (!is_numeric($price)) {
            $price = 0;
        }
        return (int) round($price * 100);
    }

    /**
	 * Get instance
	 * @return Widget
	 */
	public static function getInstance()
	{
		if (null === self::$instance) {
			self::$instance = new self();
		}
		return self::$instance;
	}
}
