<?php

/**
 * Copyright (c) HeidiPay.
 */

namespace HeidiPay\Controllers;

use Exception;
use HeidiPay\Classes\API;
use HeidiPay\Classes\Filter;
use HeidiPay\Classes\Helper;
use HeidiPay\Classes\Log;
use HeidiPay\Classes\Notice;
use HeidiPay\Classes\Widget;
use HeidiPay\Classes\Webhook as WebhookClass;
use WC_Admin_Settings;
use WC_Payment_Gateway;

class Gateway extends WC_Payment_Gateway {

    /** @var array COUNTRY_LIST */
    const COUNTRY_LIST = [
        'IT' => 'Italy',
        'CH' => 'Switzerland',
        'FR' => 'France',
        'ES' => 'Spain',
        'DE' => 'Germany',
        'UK' => 'United Kingdom'
    ];

    /** @var int MAX_AMOUNT */
    const MAX_AMOUNT = 5000;

    /** @var int MIN_AMOUNT */
    const MIN_AMOUNT = 60;

    /** @var int MAX_PRO_AMOUNT */
    const MAX_PRO_AMOUNT = 5000;

    /** @var int MIN_PRO_AMOUNT */
    const MIN_PRO_AMOUNT = 100;

    /** @var array $supports */
    public $supports = ['products', 'refunds'];

    /** @var bool $sandbox */
    public $sandbox;

    /** @var string $heidi_secret_key */
    public $heidi_secret_key;

    /**
     * Constructor.
     * @return void 
     */
    public function __construct()
    {
        $this->id                 = 'heylight';
        $this->title              = __('HeyLight', 'heylight');
        $this->method_title       = __('HeyLight', 'heylight');
        $this->has_fields         = true;
        $this->method_description = __('Illumina le tue vendite con il pagamento a rate: dilazione e finanziamento digitale.', 'heylight');
        $this->initFormFields();

        if (Widget::IS_RESTRICTED) {
            $this->title          = __('Con HeyLight dividi l’importo fino a %d rate senza interessi', 'heylight');
            $this->description    = __("HeyLight è una dilazione del pagamento a rate, senza interessi; dividi l’importo dei tuoi acquisti in modo flessibile. Sorridi ora. Paga dopo.", 'heylight');
        } else {
            $this->title          = $this->get_option('title');
            $this->description    = $this->get_option('description');
        }
        $this->enabled            = $this->get_option('enabled');
        $this->sandbox            = $this->get_option('sandbox');
        $this->heidi_secret_key   = $this->get_option('heidi_secret_key');
        
        $color = (!empty($this->get_option('widget_color')) ? $this->get_option('widget_color') : Widget::DEFAULT_WIDGET_VALUES['widget_color']);
        switch ($color) {
            case 'black':
                $this->icon = 'https://storage.googleapis.com/heidi-public-images/heylight_assets/heylight_pill_white_on_black_24px.svg';
                break;
            case 'white':
                $this->icon = 'https://storage.googleapis.com/heidi-public-images/heylight_assets/heylight_pill_black_on_white_24px.svg';
                break;
            case 'red':
            default:
                $this->icon = 'https://storage.googleapis.com/heidi-public-images/heylight_assets/heylight_pill_white_on_red_24px.svg';
                break;
        }
        // Listen for the heidipay redirect callbacks
        add_action('woocommerce_api_heidipay_success', [$this, 'success_webhook']);
        add_action('woocommerce_api_heidipay_fail', [$this, 'fail_webhook']);

        // Listen for the admin configuration updating
        add_action(
            'woocommerce_update_options_payment_gateways_' . $this->id,
            [$this, 'process_admin_options']
        );
    }

    /**
     * Process Payment
     * @param int $orderId
     * @return array
     */
    public function process_payment($orderId)
    {
        global $woocommerce;
        $order = wc_get_order($orderId);

        /** Check if order filter is PRO  */
        $terms = Filter::getOrderAllowedTerms($order);
        $filter = Filter::getOrderAllowedTerms($order, true);
        $bnplProduct = null;
        if (! empty($filter)) {
            $bnplProductId = $filter[Filter::META_PREFIX.'_product'];
            foreach (Filter::PRODUCT_TYPES as $productType) {
                if ($productType['id'] == $bnplProductId) {
                    $bnplProduct = $productType;
                    break;
                }
            }
        } else {
            $bnplProduct = Widget::getGeneralBNPLProduct();
            if (empty($bnplProduct)) {
                $types = Filter::PRODUCT_TYPES;
                $bnplProduct = reset($types);
            }
        }
        $isPro = (bool) ($bnplProduct && ! empty($bnplProduct['pro']) ? $bnplProduct['pro'] : false);
        if ($isPro && ! Widget::isProAvailable()) {
            $isPro = false;
        }

        /** Check amount */
        $total = (float) $order->get_total();
        $totalAmount = number_format($total, 2, '.', '');
        if (! Helper::checkAmount($totalAmount, $isPro, $terms)) {
            $isPro = !$isPro;
            if (! Helper::checkAmount($totalAmount, $isPro, $terms)) {
                Notice::getInstance()->addError(__('La transazione è stata rifiutata. L’importo dell’ordine non è consentito dalla configurazione di HeyLight.', 'heylight'));
                return [
                    'result'   => 'fail',
                    'redirect' => '',
                ];
            }
        }
        if ($isPro && ! Widget::isProAvailable()) {
            Notice::getInstance()->addError(__('La transazione è stata rifiutata. La versione Finanziamento digitale di HeyLight non è disponibile qui.', 'heylight'));
            return [
                'result'   => 'fail',
                'redirect' => '',
            ];
        }

        /** Check token access */
        $token = API::getAuthToken($isPro);
        if (!$token) {
            Notice::getInstance()->addError(__('La transazione è stata rifiutata. È impossibile unirsi all’API HeyLight con la configurazione attuale.', 'heylight'));
            return [
                'result'   => 'fail',
                'redirect' => '',
            ];
        }

        /** Init transaction */
        $data = Helper::getInitData($order, $token, $this);
        if (! $data) {
            Notice::getInstance()->addError(__('La transazione è stata rifiutata. Al momento è impossibile abbinare l’importo a un intervallo di pagamento HeyLight.', 'heylight'));
            return [
                'result'   => 'fail',
                'redirect' => '',
            ];
        }
        $result = API::initTransaction($data, $token, $order);
        if (empty($result['redirect_url']) || empty($result['external_contract_uuid'])) {
            $error = (! empty($result['contents']) ? json_decode($result['contents'], true) : null);
            if ($error && ! empty($error['detail']) && str_contains($error['detail'], 'Returning customer has changed core details')) {
                $message = __('Your e-mail has already been associated with another Heylight (previously PagoLight) customer name. Please verify that your name is exactly the same (including any middle names) as provided for previous payments with HeyLight. Otherwise, please use a different e-mail address or contact support@heylight.com.', 'heylight');
            } else {
                $message = __('La transazione è stata rifiutata. Impossibile reindirizzarti. Si prega di riprovare più tardi', 'heylight');
            }
            Notice::getInstance()->addError($message);
            return [
                'result'   => 'fail',
                'message'  => $message,
                'redirect' => '',
            ];
        }

        /** Save transaction ID */
        $order->set_transaction_id($result['external_contract_uuid']);
        $order->update_status('pending', __('Awaiting inline payment', 'heylight'));
        update_post_meta($orderId, WebhookClass::META_PREFIX.'_pro', $isPro);

        /** Empty cart */
        $woocommerce->cart->empty_cart();
        return [
            'result'   => 'success',
            'redirect' => $result['redirect_url'],
        ];
    }

    /**
     * Can refund order
     * @param WC_Order $order 
     * @return bool 
     */
    public function can_refund_order($order)
    {
        if (! parent::can_refund_order($order)) {
            return false;
        }

        /** Check if order contains transaction id */
        $externalContractUuid = $order->get_transaction_id();
        if (empty($externalContractUuid)) {
            return false;
        }
        return true;
    }

    /**
     * Process refund
     * @param int $orderId 
     * @param float|null $amount 
     * @param string $reason 
     * @return bool 
     */
    public function process_refund($orderId, $amount = null, $reason = '')
    {
        $order = wc_get_order($orderId);
        if (!$this->can_refund_order($order)) {
            return false;
        }

        /** Request refund */
        $externalContractUuid = $order->get_transaction_id();
        $currency = $order->get_currency();
        $isPro = (bool) get_post_meta($orderId, WebhookClass::META_PREFIX.'_pro', true);
        $result = API::requestRefund($externalContractUuid, (float) $amount, $currency, $isPro, $order);
        if (!$result) {
            return false;
        }

        /** Add order note if successful */
        $order->add_order_note(
            /* translators: 1: Refund amount, 2: Refund reason */
            sprintf(__('Refunded %1$s - Reason: %2$s', 'heylight'), $amount, $reason)
        );
        return true;
    }

    /**
     * Success webhook
     * @return void
     */
    public function success_webhook()
    {
        $order = wc_get_order((int) $_GET['id']);
        $externalContractUuid = $order->get_transaction_id();
        $isPro = (bool) get_post_meta((int) $_GET['id'], WebhookClass::META_PREFIX.'_pro', true);
        $token = API::getAuthToken($isPro);
        if (!$token || !$externalContractUuid || !API::checkOrderStatus($token, $externalContractUuid, $order)) {
            $error = __('Siamo spiacenti, ma si è verificato un problema nel completamento dell\'ordine. Si prega di contattare il venditore', 'heylight');
            Log::getInstance()->debug('Redirection after order status checking failure', $order);
            Notice::getInstance()->addError($error);
            wp_redirect(wc_get_checkout_url());
        }

        Log::getInstance()->debug('Redirection after payment success', $order);
        $order->payment_complete();
        wp_redirect($this->get_return_url($order));
    }

    /**
     * Fail webhook
     * @return void
     */
    public function fail_webhook()
    {
        global $woocommerce;
        $order = wc_get_order($_GET['id']);
        WC()->session->set('order_awaiting_payment', false);
        $order->update_status('cancelled');
        $order->add_order_note('Payment Failed');
        $woocommerce->cart->empty_cart();

        // add cart contents
        foreach ($order->get_items() as $item) {
            /** @var WC_Order_Item_Product $item */
            $id  = $item->get_product_id();
            $variationId = $item->get_variation_id();
            $qty = $item->get_quantity();
            $woocommerce->cart->add_to_cart($id, $qty, $variationId);
        }

        Log::getInstance()->debug('Redirection after payment failure', $order);
        Notice::getInstance()->addError(__('La transazione è stata rifiutata. Si prega di riprovare più tardi', 'heylight'));
        wp_redirect($order->get_cancel_order_url());
    }

    /**
     * Is available
     * @return bool
     */
    public function is_available()
    {
        /** Check enabled */
        $is_available = parent::is_available();
        if (! $is_available || ! $this->enabled || Widget::PPM_ONLY) {
            return false;
        }

        /** Check currency */
        if (! empty(Widget::RESTRICTED_CURRENCY) && get_woocommerce_currency() !== Widget::RESTRICTED_CURRENCY) {
            return false;
        }

        /** Special check for backoffice orders */
        if (! empty($_GET['pay_for_order'])) {
            global $wp;
            $orderId = $wp->query_vars['order-pay'];
            $order = wc_get_order($orderId);
            if (! empty($order) && $order) {
                return $this->isAvailableForOrder($order);
            }
        }

        /** Check backend */
        if (is_admin()) {
            return true;
        }

        /** Check cart */
        if (!WC()->cart || ! did_action('wp_loaded')) {
            return true;
        }

        /** Check country */
        if (Helper::isCountryRestricted()) {
            return false;
        }
        
        /** Check depend on promotional message */
        if ($this->get_option('widget_enabled_cart') === 'yes' && ! Filter::shouldCartDisplayWidget(WC()->cart)) {
            return false;
        }

        /** Check if cart filter is PRO */
        $terms = Filter::getCartAllowedTerms(WC()->cart);
        $filter = Filter::getCartAllowedTerms(WC()->cart, true);
        $bnplProduct = null;
        if (! empty($filter)) {
            $bnplProductId = $filter[Filter::META_PREFIX.'_product'];
            foreach (Filter::PRODUCT_TYPES as $productType) {
                if ($productType['id'] == $bnplProductId) {
                    $bnplProduct = $productType;
                    break;
                }
            }
        } else {
            $bnplProduct = Widget::getGeneralBNPLProduct();
            if (empty($bnplProduct)) {
                $types = Filter::PRODUCT_TYPES;
                $bnplProduct = reset($types);
            }
        }
        $isPro = (bool) ($bnplProduct && ! empty($bnplProduct['pro']) ? $bnplProduct['pro'] : false);
        if ($isPro && ! Widget::isProAvailable()) {
            $isPro = false;
        } else if (! $isPro && ! Widget::isRegularAvailable()) {
            $isPro = true;
        }

        /** Get terns */
        if ($terms == null) {
            $this->init_settings();
            $terms = (! empty($this->settings['widget_terms']) ? (is_array($this->settings['widget_terms']) ? $this->settings['widget_terms'] : json_decode($gateway->settings['widget_terms'])) : Widget::DEFAULT_WIDGET_VALUES['widget_terms']);
        }

        /** Check amount */
        $cartTotal = (wc_prices_include_tax() ? WC()->cart->get_cart_contents_total() + WC()->cart->get_cart_contents_tax() : WC()->cart->get_cart_contents_total());
        $totalAmount = number_format($cartTotal, 2, '.', '');
        if (! Helper::checkAmount($totalAmount, $isPro, $terms)) {
            $isPro = !$isPro;
            if (! Helper::checkAmount($totalAmount, $isPro, $terms)) {
                return false;
            }
        }
        if ($isPro && ! Widget::isProAvailable()) {
            return false;
        } else if (! $isPro && ! Widget::isRegularAvailable()) {
            return false;
        }

        if ($isPro) {
            $minimumInstalmentPrice = (! empty($this->settings['widget_min_instalment_pro']) ? (float) $this->settings['widget_min_instalment_pro'] : Widget::DEFAULT_WIDGET_VALUES['widget_min_instalment_pro']);
        } else {
            $minimumInstalmentPrice = (! empty($this->settings['widget_min_instalment']) ? (float) $this->settings['widget_min_instalment'] : Widget::DEFAULT_WIDGET_VALUES['widget_min_instalment']);
        }
        $allowedTerms = Helper::getAllowedTerms($terms, (int) round($totalAmount * 100, 0), $minimumInstalmentPrice);
        if (empty($allowedTerms)) {
            return false;
        }

        /** Change logo + text to PRO */
        if ($isPro) {
            $this->title          = (! empty($this->get_option('title_pro')) ? $this->get_option('title_pro') : __('Con HeyLight paghi un po’ per volta, fino a %d rate', 'heylight'));
            $this->description    = (! empty($this->get_option('description_pro')) ? $this->get_option('description_pro') : __('HeyLight è un finanziamento digitale che permette di pagare a rate online. Scopri le condizioni e realizza i tuoi acquisti.', 'heylight'));
            $learnMoreLink = Widget::getHeidiDomain();
            $learnMoreLink .= '/footer/';
            $learnMoreLink .= '?key='.urlencode($this->settings['widget_api_key']);
            $learnMoreLink .= '&amount='.(int) round($totalAmount * 100, 0);
            $learnMoreLink .= '&terms='.urlencode(implode(',', $allowedTerms));
            $code = (! empty($filter) ? $filter[Filter::META_PREFIX.'_code'] : (! empty($this->settings['CODE']) ? $this->settings['CODE'] : '-'));
            $learnMoreLink .= '&code='.urlencode($code);
            $this->description = str_replace('%learn_more_url%', $learnMoreLink, $this->description);
        }
        $this->title = str_replace('0%', '0%%', $this->title);
        $this->title = sprintf($this->title, max($allowedTerms));

        return true;
    }

    /**
     * Is available for order
     * @param WC_Order $order
     * @return bool
     */
    public function isAvailableForOrder($order)
    {
        if (! $order || ! is_a($order, 'WC_Order')) {
            return false;
        }

        /** Check if order filter is PRO */
        $terms = Filter::getOrderAllowedTerms($order);
        $filter = Filter::getOrderAllowedTerms($order, true);
        $bnplProduct = null;
        if (! empty($filter)) {
            $bnplProductId = $filter[Filter::META_PREFIX.'_product'];
            foreach (Filter::PRODUCT_TYPES as $productType) {
                if ($productType['id'] == $bnplProductId) {
                    $bnplProduct = $productType;
                    break;
                }
            }
        } else {
            $bnplProduct = Widget::getGeneralBNPLProduct();
            if (empty($bnplProduct)) {
                $types = Filter::PRODUCT_TYPES;
                $bnplProduct = reset($types);
            }
        }
        $isPro = (bool) ($bnplProduct && ! empty($bnplProduct['pro']) ? $bnplProduct['pro'] : false);
        if ($isPro && ! Widget::isProAvailable()) {
            $isPro = false;
        } else if (! $isPro && ! Widget::isRegularAvailable()) {
            $isPro = true;
        }

        /** Get terns */
        if ($terms == null) {
            $this->init_settings();
            $terms = (! empty($this->settings['widget_terms']) ? (is_array($this->settings['widget_terms']) ? $this->settings['widget_terms'] : json_decode($gateway->settings['widget_terms'])) : Widget::DEFAULT_WIDGET_VALUES['widget_terms']);
        }

        /** Check amount */
        $totalAmount = number_format($order->get_total(), 2, '.', '');
        if (! Helper::checkAmount($totalAmount, $isPro, $terms)) {
            $isPro = !$isPro;
            if (! Helper::checkAmount($totalAmount, $isPro, $terms)) {
                return false;
            }
        }
        if ($isPro && ! Widget::isProAvailable()) {
            return false;
        } else if (! $isPro && ! Widget::isRegularAvailable()) {
            return false;
        }

        if ($isPro) {
            $minimumInstalmentPrice = (! empty($this->settings['widget_min_instalment_pro']) ? (float) $this->settings['widget_min_instalment_pro'] : Widget::DEFAULT_WIDGET_VALUES['widget_min_instalment_pro']);
        } else {
            $minimumInstalmentPrice = (! empty($this->settings['widget_min_instalment']) ? (float) $this->settings['widget_min_instalment'] : Widget::DEFAULT_WIDGET_VALUES['widget_min_instalment']);
        }
        $allowedTerms = Helper::getAllowedTerms($terms, (int) round($totalAmount * 100, 0), $minimumInstalmentPrice);
        if (empty($allowedTerms)) {
            return false;
        }

        /** Change logo + text to PRO */
        if ($isPro) {
            $this->title          = (! empty($this->get_option('title_pro')) ? $this->get_option('title_pro') : __('Con HeyLight paghi un po’ per volta, fino a %d rate', 'heylight'));
            $this->description    = (! empty($this->get_option('description_pro')) ? $this->get_option('description_pro') : __('HeyLight è un finanziamento digitale che permette di pagare a rate online. Scopri le condizioni e realizza i tuoi acquisti.', 'heylight'));
            $learnMoreLink = Widget::getHeidiDomain();
            $learnMoreLink .= '/footer/';
            $learnMoreLink .= '?key='.urlencode($this->settings['widget_api_key']);
            $learnMoreLink .= '&amount='.(int) round($totalAmount * 100, 0);
            $learnMoreLink .= '&terms='.urlencode(implode(',', $allowedTerms));
            $code = (! empty($filter) ? $filter[Filter::META_PREFIX.'_code'] : (! empty($this->settings['CODE']) ? $this->settings['CODE'] : '-'));
            $learnMoreLink .= '&code='.urlencode($code);
            $this->description = str_replace('%learn_more_url%', $learnMoreLink, $this->description);
        }
        $this->title = str_replace('0%', '0%%', $this->title);
        $this->title = sprintf($this->title, max($allowedTerms));

        return true;
    }

    /**
     * Initialise HeidiPay Gateway Settings Form Fields.
     * @return void
     */
    public function initFormFields()
    {
        $this->form_fields = [
            'enabled'         => [
                'title'       => __('Plugin attivo', 'heylight'),
                'label'       => __('Abilitare', 'heylight'),
                'type'        => 'checkbox',
                'description' => '',
                'default'     => 'no',
            ],
            
            'checkout'        => [
                'title'       => __('Chiavi di attivazione', 'heylight'),
                'type'        => 'separator'
            ],
            'heidi_secret_key' => [
                'title'   => __('Chiave API per HeyLight Dilazione', 'heylight'),
                'type'    => 'text',
                'default' => '',
            ],
            'heidi_secret_key_pro' => [
                'title'   => __('Chiave API per HeyLight Finanziamento digitale', 'heylight'),
                'type'    => 'text',
                'default' => '',
            ],
            'widget_api_key' => [
                'title'   => __('Public Key', 'heylight'),
                'type'    => 'text',
                'default' => '',
            ],
            'sandbox'         => [
                'title'       => __('Modalità di prova', 'heylight'),
                'label'       => __('Spunta la casella per abilitare HeyLight in modalità di test', 'heylight'),
                'type'        => 'checkbox',
                'description' => '',
                'default'     => 'yes',
            ],

            'text_nbpl'        => [
                'title'       => __('', 'heylight'),
                'type'        => 'separator'
            ],
            'title'           => [
                'title'       => __('', 'heylight'),
                'type'        => 'text',
                'description' => __('This controls the title which the user sees during checkout.', 'heylight'),
                'default'     => __("Con HeyLight dividi l’importo fino a %d rate senza interessi", 'heylight'),
            ],
            /*'checkout_image_url'  => [
                'title'       => __('Checkout icon URL', 'heylight'),
                'type'        => 'text',
                'description' => __('This icon to display next to the checkout option.', 'heylight'),
                'default'     => 'https://storage.googleapis.com/heidi-public-images/heylight_assets/heylight_pill_white_on_red.svg',
            ],*/
            'description'     => [
                'title'       => __('Description', 'heylight'),
                'type'        => 'text',
                'description' => __(
                    'The description displayed when this payment method is selected.',
                    'heylight'
                ),
                'default'     => '',
            ],

            'text_pro'        => [
                'title'       => __('', 'heylight'),
                'type'        => 'separator'
            ],
            'title_pro'           => [
                'title'       => __('', 'heylight'),
                'type'        => 'text',
                'description' => __('This controls the title which the user sees during checkout.', 'heylight'),
                'default'     => __("Con HeyLight paghi un po’ per volta, fino a %d rate", 'heylight'),
            ],
            /*'checkout_image_url_pro'  => [
                'title'       => __('', 'heylight'),
                'type'        => 'text',
                'description' => __('This icon to display next to the checkout option.', 'heylight'),
                'default'     => 'https://storage.googleapis.com/heidi-public-images/heylight_assets/heylight_pill_white_on_red.svg',
            ],*/
            'description_pro'     => [
                'title'       => __('', 'heylight'),
                'type'        => 'text',
                'description' => __(
                    'The description displayed when this payment method is selected.',
                    'heylight'
                ),
                'default'     => 'HeyLight è un finanziamento digitale che permette di pagare a rate online. Scopri le condizioni e realizza i tuoi acquisti.',
            ],
            

            'ppm'            => [
                'title'   => __('Impostazioni checkout e messaggi promozionali', 'heylight'),
                'type'    => 'separator'
            ],
            'widget_color' => [
                'title' => __('Scegli il colore del logo HeyLight', 'heidipay'),
                'type' => 'select',
                'default' => 'red',
                'options' => [
                    'red' => __('Default', 'heidipay'),
                    'black' => __('Nero', 'heidipay'),
                    'white' => __('Bianco', 'heidipay'),
                ],
            ],
            'widget_enabled' => [
                'title'   => __('Mostra il messaggio promozionale nella pagina di prodotto', 'heylight'),
                'label'   => __('Abilitare', 'heylight'),
                'type'    => 'checkbox',
                'default' => (Widget::DEFAULT_WIDGET_VALUES['widget_enabled'] ? 'yes' : 'no'),
            ],
            'widget_enabled_cart' => [
                'title'   => __('Mostra il messaggio promozionale sul carrello/checkout', 'heylight'),
                'label'   => __('Abilitare', 'heylight'),
                'type'    => 'checkbox',
                'default' => (Widget::DEFAULT_WIDGET_VALUES['widget_enabled_cart'] ? 'yes' : 'no'),
            ],


            'general_settings' => [
                'title'   => __('Impostazioni generali', 'heylight'),
                'type'    => 'separator'
            ],
            'order_ref_sent' => [
                'title' => __('Order reference sent', 'heidipay'),
                'type' => 'select',
                'default' => 'id',
                'options' => [
                    'id' => __('Order ID', 'heidipay'),
                    'reference' => __('Order reference', 'heidipay'),
                ],
            ],
            'widget_fee' => [
                'title'   => __('N/A', 'heylight'),
                'type'    => 'text',
                'default' => Widget::DEFAULT_WIDGET_VALUES['widget_fee'],
            ],
            'PRODUCT' => [
                'title' => __('Seleziona prodotto', 'heylight'),
                'type' => 'select',
                'default' => Widget::DEFAULT_WIDGET_VALUES['PRODUCT'],
                'options' => $this->getProductOptions(),
            ],
            'widget_terms' => [
                'title'   => __('Seleziona il numero di rate', 'heylight'),
                'type'    => 'multicheckbox',
                'default' => Widget::DEFAULT_WIDGET_VALUES['widget_terms'],
                'options' => $this->getTermOptions(),
            ],
            'CODE' => [
                'title' => __('Codice tabella (solo per HeyLight Finanziamento)', 'heylight'),
                'type' => 'text',
                'default' => Widget::DEFAULT_WIDGET_VALUES['CODE'],
            ],

            'std_section'     => [
                'title'   => __('HeyLight Dilazione', 'heylight'),
                'type'    => 'separator'
            ],
            'widget_min_amount' => [
                'title'   => __('Attiva il prodotto HeyLight solo per i prodotti o gli ordini a partire da', 'heylight'),
                'type'    => 'text',
                'default' => Widget::DEFAULT_WIDGET_VALUES['widget_min_amount'],
            ],
            'widget_max_amount' => [
                'title'   => __('Attiva il prodotto HeyLight solo per i prodotti o gli ordini fino a', 'heylight'),
                'type'    => 'text',
                'default' => Widget::DEFAULT_WIDGET_VALUES['widget_max_amount'],
            ],
            'widget_min_instalment' => [
                'title'   => __('Visualizza messaggio solo se l’importo di ciascuna rata è maggiore di', 'heylight'),
                'type'    => 'text',
                'default' => Widget::DEFAULT_WIDGET_VALUES['widget_min_instalment'],
            ],
            'widget_with_card' => [
                'title'   => __('', 'heylight'),
                'type'    => 'checkbox',
                'default' => (Widget::DEFAULT_WIDGET_VALUES['widget_with_card'] ? 'yes' : 'no'),
            ],

            'pro_section'    => [
                'title'   => __('HeyLight Finanziamento digitale', 'heylight'),
                'type'    => 'separator'
            ],
            'widget_min_amount_pro' => [
                'title'   => __('Attiva il prodotto HeyLight Finanziamento digitale solo per i prodotti o gli ordini a partire da', 'heylight'),
                'type'    => 'text',
                'default' => Widget::DEFAULT_WIDGET_VALUES['widget_min_amount_pro'],
            ],
            'widget_max_amount_pro' => [
                'title'   => __('Attiva il prodotto HeyLight Finanziamento digitale solo per i prodotti o gli ordini fino a', 'heylight'),
                'type'    => 'text',
                'default' => Widget::DEFAULT_WIDGET_VALUES['widget_max_amount_pro'],
            ],
            'widget_min_instalment_pro' => [
                'title'   => __('Visualizza messaggio solo se l’importo di ciascuna rata è maggiore di', 'heylight'),
                'type'    => 'text',
                'default' => Widget::DEFAULT_WIDGET_VALUES['widget_min_instalment_pro'],
            ],

            'geoloc'      => [
                'title'   => __('Geolocalizzazione', 'heylight'),
                'type'    => 'separator'
            ],
            'countries'  => [
                'title'    => __('Mostra il prezzo al mese ai clienti di:', 'heylight'),
                'type' => 'multiselect',
                'default' => [],
                'options' => self::COUNTRY_LIST,
            ],

            /*
            'portal'      => [
                'title'   => __('Impostazioni Aggiuntive', 'heylight'),
                'type'    => 'separator'
            ],
            'portal_link'  => [
                'title'    => __('Per ulteriori impostazioni, configurazioni e dati, visita il %link%', 'heylight'),
                'link'     => [
                    'name' => __('Merchant Portal', 'heylight'),
                    'url'  => 'https://merchant-portal.heidipay.com/HeyLight'
                ],
                'type' => 'doclink'
            ],
            */

            'category_title'    => [
                'title' => __('Opzioni delle categorie', 'heylight'),
                'type'  => 'separator'
            ],
            'categories' => [
                'title' => __('Opzioni delle categorie', 'heylight'),
                'type' => 'categories'
            ]
        ];
        
        $maxMindSet = Helper::isMaxMindSet();
        foreach ($this->form_fields as $field => $option) {
            if (! Widget::CAN_BE_PRO && in_array($field, ['heidi_secret_key_pro', 'PRODUCT', 'CODE', 'widget_min_amount_pro', 'widget_max_amount_pro', 'widget_min_instalment_pro', 'pro_section', 'title_pro', 'checkout_image_url_pro', 'description_pro', 'text_pro'])) {
                $this->form_fields[$field] = null;
                unset($this->form_fields[$field]);
                continue;
            }

            if (! $maxMindSet && in_array($field,  ['geoloc', 'countries'])) {
                $this->form_fields[$field] = null;
                unset($this->form_fields[$field]);
                continue;
            }

            if (! Widget::PRO_NEED_CODE && $field == 'CODE') {
                $this->form_fields[$field] = null;
                unset($this->form_fields[$field]);
                continue;
            }

            if (! Widget::CAN_DISABLE_QR && $field == 'widget_with_card') {
                $this->form_fields[$field] = null;
                unset($this->form_fields[$field]);
                continue;
            }

            if ($field == 'widget_color' && false) {
                $this->form_fields[$field] = null;
                unset($this->form_fields[$field]);
                continue;
            }

            if ($field == 'order_ref_sent' && true) {
                $this->form_fields[$field] = null;
                unset($this->form_fields[$field]);
                continue;
            }

            if (Widget::IS_RESTRICTED) {
                if (in_array($field, ['title', 'checkout_image_url', 'description', 'title_pro', 'checkout_image_url_pro', 'description_pro', 'text_nbpl', 'text_pro', 'widget_fee'])) {
                    $this->form_fields[$field] = null;
                    unset($this->form_fields[$field]);
                    continue;
                }

                if (in_array($field, Widget::RESTRICTED_VALUES) && $field !== 'widget_min_amount') {
                    $this->form_fields[$field]['custom_attributes'] = ['readonly' => 'readonly'];
                } else if ($field === 'widget_min_amount') {
                    $this->form_fields[$field]['description'] = __('Importo minimo', 'heylight').' '.self::MIN_AMOUNT;
                } else if ($field === 'widget_min_amount_pro') {
                    $this->form_fields[$field]['description'] = __('Importo minimo', 'heylight').' '.self::MIN_PRO_AMOUNT;
                }
            }

            if (Widget::PPM_ONLY && in_array($field, ['title', 'checkout_image_url', 'description', 'title_pro', 'checkout_image_url_pro', 'description_pro', 'heidi_secret_key', 'heidi_secret_key_pro', 'text_nbpl', 'text_pro', 'sandbox'])) {
                $this->form_fields[$field] = null;
                unset($this->form_fields[$field]);
                continue;
            }
        }
    }

    /**
     * Process admin options
     * @return bool 
     */
    public function process_admin_options()
    {
        $this->init_settings();
        $post_data = $this->get_post_data();
        foreach ($this->get_form_fields() as $key => $field) {
            if ('title' !== $this->get_field_type($field)) {
                try {
                    if (Widget::IS_RESTRICTED && in_array($key, Widget::RESTRICTED_VALUES) && $key !== 'widget_min_amount' && $key !== 'widget_min_amount_pro') {
                        $this->settings[$key] = Widget::DEFAULT_WIDGET_VALUES[$key];
                        continue;
                    } else if (Widget::IS_RESTRICTED && $key === 'widget_min_amount') {
                        $value = $this->get_field_value($key, $field, $post_data);
                        if ((float) $value >= self::MIN_AMOUNT) {
                            $this->settings[$key] = $value;
                        } else {
                            WC_Admin_Settings::add_error(__('Importo minimo', 'heylight').' '.self::MIN_AMOUNT);
                        }
                        continue;
                    } else if (Widget::IS_RESTRICTED && $key === 'widget_min_amount_pro') {
                        $value = $this->get_field_value($key, $field, $post_data);
                        if ((float) $value >= self::MIN_PRO_AMOUNT) {
                            $this->settings[$key] = $value;
                        } else {
                            WC_Admin_Settings::add_error(__('Importo minimo', 'heylight').' '.self::MIN_PRO_AMOUNT);
                        }
                        continue;
                    } else if ($key === 'categories') {
                        $this->saveCategories($post_data);
                        continue;
                    }
                    $this->settings[$key] = $this->get_field_value($key, $field, $post_data);
                } catch (Exception $e) {
                    $this->add_error($e->getMessage());
                }
            }
        }

        $fee = (float) (! empty($this->settings['widget_fee']) ? $this->settings['widget_fee'] : Widget::DEFAULT_WIDGET_VALUES['widget_fee']);
        $product = (string) (! empty($this->settings['PRODUCT']) ? $this->settings['PRODUCT'] : Widget::DEFAULT_WIDGET_VALUES['PRODUCT']);
        if ($product === 'HEYLIGHT_FINANCING_CH' && $fee > 0)  {
            $this->settings['widget_fee'] = 0;
        }
        
        return update_option($this->get_option_key(), apply_filters('woocommerce_settings_api_sanitized_fields_' . $this->id, $this->settings), 'yes');
    }

    /**
     * Get option from DB.
     *
     * Gets an option from the settings API, using defaults if necessary to prevent undefined notices.
     *
     * @param  string $key Option key.
     * @param  mixed  $empty_value Value when empty.
     * @return string The value specified for the option or a default value for the option.
     */
    public function get_option($key, $empty_value = null)
    {
        if (!Widget::IS_RESTRICTED) {
            return parent::get_option($key, $empty_value);
        }

        if (empty($this->settings)) {
            $this->init_settings();
        }

        if (in_array($key, Widget::RESTRICTED_VALUES) && $key !== 'widget_min_amount' && $key !== 'widget_min_amount_pro') {
            return Widget::DEFAULT_WIDGET_VALUES[$key];
        } else if ($key === 'widget_min_amount') {
            $value = (float) parent::get_option($key, $empty_value);
            if ((float) $value >= self::MIN_AMOUNT) {
                return $value;
            } else {
                return self::MIN_AMOUNT;
            }
        } else if ($key === 'widget_min_amount_pro') {
            $value = (float) parent::get_option($key, $empty_value);
            if ((float) $value >= self::MIN_PRO_AMOUNT) {
                return $value;
            } else {
                return self::MIN_PRO_AMOUNT;
            }
        }
        return parent::get_option($key, $empty_value);
    }

    /**
     * Validate widget_terms field
     * @param mixed $value 
     * @return array|string 
     */
    public function validate_widget_terms_field($key, $value)
    {
        $values = is_array($value) ? array_map('wc_clean', array_map('stripslashes', $value)) : '';
        if (!empty($values) && is_array($values) && count($value) > Filter::MAX_INSTALMENTS) {
            WC_Admin_Settings::add_error(__('Maximum 6 choices for the number of instalments', 'heylight'));
            throw new Exception(__('Maximum 6 choices for the number of instalments', 'heylight'));
        }
        return $values;
    }

    /**
     * Validate widget_max_amount field
     * @param mixed $value 
     * @return array|string 
     */
    public function validate_widget_max_amount_field($key, $value)
    {
        if (! empty($value) && intval($value) > self::MAX_AMOUNT) {
            WC_Admin_Settings::add_error(__('Importo massimo', 'heylight').' '.self::MAX_AMOUNT);
            throw new Exception(__('Importo massimo', 'heylight').' '.self::MAX_AMOUNT);
        }
        return $value;
    }

    /**
     * Validate widget_max_amount_pro field
     * @param mixed $value 
     * @return array|string 
     */
    public function validate_widget_max_amount_pro_field($key, $value)
    {
        if (! empty($value) && intval($value) > self::MAX_PRO_AMOUNT) {
            WC_Admin_Settings::add_error(__('Importo massimo', 'heylight').' '.self::MAX_PRO_AMOUNT);
            throw new Exception(__('Importo massimo', 'heylight').' '.self::MAX_PRO_AMOUNT);
        }
        return $value;
    }

    /**
     * Generate multicheckbox HTML
     * @param string $key 
     * @param mixed $data 
     * @return string
     */
    public function generate_multicheckbox_html($key, $data)
    {
        $this->registerAssets();
        wp_enqueue_style('heidipay_filter-styles');
        wp_enqueue_script('heidipay_filter-scripts');
        $field    = $this->plugin_id . $this->id . '_' . $key;
        $defaults = [
            'class'             => 'multicheckbox',
            'css'               => '',
            'custom_attributes' => [],
            'desc_tip'          => false,
            'description'       => '',
            'title'             => '',
        ];
        $data = wp_parse_args($data, $defaults);

        ob_start();
        ?>
        <tr valign="top" class="multicheckbox">
            <th scope="row" class="titledesc">
                <label for="<?php echo esc_attr($field); ?>"><?php echo wp_kses_post($data['title']); ?></label>
            </th>
            <td class="forminp">
                <fieldset class="instalments-checkboxes">
                    <?php foreach ($data['options'] as $value => $name) : ?>
                        <div class="instalment-checkbox">
                            <input type="checkbox" id="<?php echo esc_attr($field) . '_' . $value; ?>"
                                value="<?php echo $value; ?>"
                                data-pro="<?php echo (in_array($value, Widget::PRO_TERMS) ? 'true' : 'false'); ?>"
                                name="<?php echo esc_attr($field); ?>[]"
                                <?php echo ($this->settings[$key] != null && in_array($value, $this->settings[$key]) ? 'checked="checked"' : ''); ?>/>
                            <label for="<?php echo esc_attr($field) . '_' . $value; ?>" title="<?php echo $name; ?>">
                                <?php echo $value; ?>
                            </label>
                        </div>
                    <?php endforeach; ?>
                    <script type="text/javascript">
                        var errorMaxInstalments = "<?php echo __('Si prega di selezionare fino a 6 scelte', 'heylight'); ?>";
                        var errorMaxAmount = "<?php echo __('The maximum amount for HeyLight Dilazione is', 'heylight').' '.self::MAX_AMOUNT; ?>";
                        var bnplProducts = <?php echo json_encode(Filter::PRODUCT_TYPES); ?>;
                        var defaultPro = <?php echo (Widget::getGeneralBNPLProduct()['pro'] ? 'true' : 'false'); ?>;
                    </script>
                </fieldset>
            </td>
        </tr>
        <?php
        return ob_get_clean();
    }

    /**
     * Generate multiselect HTML
     * @param string $key 
     * @param mixed $data 
     * @return string
     */
    public function generate_multiselect_html($key, $data)
    {
        $field    = $this->plugin_id . $this->id . '_' . $key;
        $defaults = [
            'class'             => 'multiselect',
            'css'               => '',
            'custom_attributes' => [],
            'desc_tip'          => false,
            'description'       => '',
            'title'             => '',
        ];
        $data = wp_parse_args($data, $defaults);

        ob_start();
        ?>
        <tr valign="top" class="multiselect">
            <th scope="row" class="titledesc">
                <label for="<?php echo esc_attr($field); ?>"><?php echo wp_kses_post($data['title']); ?></label>
            </th>
            <td class="forminp">
                <fieldset class="multi-select">
                    <?php foreach ($data['options'] as $value => $name) : ?>
                        <div class="option">
                            <input type="checkbox" id="<?php echo esc_attr($field) . '_' . $value; ?>"
                                value="<?php echo $value; ?>"
                                name="<?php echo esc_attr($field); ?>[]"
                                <?php echo ($this->settings[$key] != null && in_array($value, $this->settings[$key]) ? 'checked="checked"' : ''); ?>/>
                            <label for="<?php echo esc_attr($field) . '_' . $value; ?>" title="<?php echo $name; ?>">
                                <?php echo $name; ?>
                            </label>
                        </div>
                    <?php endforeach; ?>
                </fieldset>
            </td>
        </tr>
        <?php
        return ob_get_clean();
    }

    /**
     * Register assets
     * @return void 
     */
    public function registerAssets()
    {
        $pluginDirectory = realpath(dirname(__FILE__).'/../');
        $pluginUrl = plugin_dir_url($pluginDirectory);
        $assetsUrl = $pluginUrl.'assets/';
        wp_register_style(
            'heidipay_filter-styles',
            $assetsUrl.'css/admin.css'
        );
        wp_register_script(
            'heidipay_filter-scripts',
            $assetsUrl.'js/admin.js',
            ['jquery']
        );
    }

    /**
     * Generate the separator HTML
     * @param string $key 
     * @param mixed $data 
     * @return string 
     */
    public function generate_separator_html($key, $data)
    {
        $field    = $this->plugin_id . $this->id . '_' . $key;
        $defaults = [
            'class'             => 'separator',
            'css'               => '',
            'custom_attributes' => [],
            'desc_tip'          => false,
            'description'       => '',
            'title'             => '',
        ];
        $data = wp_parse_args($data, $defaults);

        ob_start();
        ?>
        <tr valign="top" class="separator" style="border-bottom: 1px solid #000000;">
            <th scope="row" class="titledesc">
                <h2 style="margin: 0;" for="<?php echo esc_attr($field); ?>"><?php echo wp_kses_post($data['title']); ?></h2>
            </th>
            <td class="forminp"></td>
        </tr>
        <?php
        return ob_get_clean();
    }

    /**
     * Generate the doclink HTML
     * @param string $key 
     * @param mixed $data 
     * @return string 
     */
    public function generate_doclink_html($key, $data)
    {
        $defaults = [
            'class'             => 'doclink',
            'css'               => '',
            'custom_attributes' => [],
            'desc_tip'          => false,
            'description'       => '',
            'title'             => '',
            'link'                => [
                'name'            => '',
                'url'            => ''
            ]
        ];
        $data = wp_parse_args($data, $defaults);

        $text = wp_kses_post($data['title']);
        if (strpos($text, '%link%') !== false && !empty($data['link']['name'])) {
            $htmlLink = '<a href="' . $data['link']['url'] . '" target="_blank">' . $data['link']['name'] . '</a>';
            $text = str_replace('%link%', $htmlLink, $text);
        }

        ob_start();
        ?>
        <tr valign="top" class="doclink">
            <th colspan="2" scope="row">
                <?php echo $text; ?>
            </th>
        </tr>
        <?php
        return ob_get_clean();
    }

    /**
     * Generate categories HTML
     * @param string $key 
     * @param mixed $data 
     * @return string
     */
    public function generate_categories_html($key, $data)
    {
        $field = $this->plugin_id . $this->id . '_' . $key;
        $categories = get_categories([
            'taxonomy' => 'product_cat',
            'orderby' => 'name',
            'show_count' => 0,
            'pad_counts' => 0,
            'hierarchical' => 1,
            'hide_empty' => 0
        ]);
        $noFilterCategories = [];
        $categoryFilters = [];
        foreach ($categories as $category) {
            $filters = Filter::getFilterForCategory($category);
            if (! $filters) {
                $noFilterCategories[] = $category;
                continue;
            }

            $categoryFilters[$category->term_id] = [
                'category' => $category,
                'filters' => $filters
            ];
        }

        ob_start();
        ?>
        <tr valign="top" class="categories">
            <td class="forminp" colspan="2">
                <table class="form-table">
                    <thead>
                        <tr>
                            <th><?php _e('Categoria', 'heylight'); ?></th>
                            <th><?php _e('Spunta per disabilitare', 'heylight'); ?></th>
                            <th><?php _e('Applica la regola al carrello misto', 'heylight'); ?></th>
                            <th><?php _e('Rate', 'heylight'); ?></th>
                            <?php if (Widget::CAN_BE_PRO) : ?>
                                <th><?php _e('Seleziona prodotto', 'heylight'); ?></th>
                                <th><?php _e('Codice tabella (solo per HeyLight Finanziamento)', 'heylight'); ?></th>
                            <?php endif; ?>
                            <th></th>
                        </tr>
                    </thead>
                    <tbody>
                    <?php foreach ($categoryFilters as $categoryId => $categoryData) : ?>
                        <tr class="category-filter">
                            <td class="category-name">
                                <?php echo $categoryData['category']->name; ?>
                            </td>

                            <td>
                                <label style="display: block;" for="<?php echo esc_attr($field).'_'.Filter::META_PREFIX.'_disable_'.$categoryId; ?>">
                                    <input type="checkbox" name="<?php echo esc_attr($field).'_'.Filter::META_PREFIX.'_disable_'.$categoryId; ?>" 
                                        id="<?php echo esc_attr($field).'_'.Filter::META_PREFIX.'_disable_'.$categoryId; ?>" 
                                        <?php echo ($categoryData['filters'][Filter::META_PREFIX.'_disable'] != null ? 'checked="checked"' : ''); ?>>
                                </label>
                            </td>

                            <td>
                                <label style="display: block;" for="<?php echo esc_attr($field).'_'.Filter::META_PREFIX.'_apply_to_basket_'.$categoryId; ?>">
                                    <input type="checkbox" name="<?php echo esc_attr($field).'_'.Filter::META_PREFIX.'_apply_to_basket_'.$categoryId; ?>" 
                                        id="<?php echo esc_attr($field).'_'.Filter::META_PREFIX.'_apply_to_basket_'.$categoryId; ?>" 
                                        <?php echo ($categoryData['filters'][Filter::META_PREFIX.'_apply_to_basket'] != null ? 'checked="checked"' : ''); ?>>
                                </label>
                            </td>

                            <td>
                                <div class="instalments-checkboxes">
                                <?php
                                $values = (is_array($categoryData['filters'][Filter::META_PREFIX.'_terms']) ? $categoryData['filters'][Filter::META_PREFIX.'_terms'] : json_decode($categoryData['filters'][Filter::META_PREFIX.'_terms']));
                                $options = Gateway::getTermOptions();
                                foreach ($options as $key => $name) {
                                    ?>
                                    <div class="instalment-checkbox">
                                        <input type="checkbox" id="<?php echo esc_attr($field).'_'.Filter::META_PREFIX.'_terms_'.$categoryId.'_'.$key; ?>"
                                            value="<?php echo $key; ?>"
                                            data-pro="<?php echo (in_array($key, Widget::PRO_TERMS) ? 'true' : 'false'); ?>"
                                            name="<?php echo esc_attr($field).'_'.Filter::META_PREFIX.'_terms_'.$categoryId.'[]'; ?>" 
                                            <?php echo (in_array($key, $values) ? 'checked="checked"' : ''); ?>/>
                                        <label for="<?php echo esc_attr($field).'_'.Filter::META_PREFIX.'_terms_'.$categoryId.'_'.$key; ?>" title="<?php echo $name; ?>">
                                            <?php echo $key; ?>
                                        </label>
                                    </div>
                                    <?php
                                }
                                ?>
                                </div>
                            </td>

                            <?php if (Widget::CAN_BE_PRO) : ?>
                                <td>
                                    <label style="display: block;" for="<?php echo esc_attr($field).'_'.Filter::META_PREFIX.'_product_'.$categoryId; ?>">
                                        <select name="<?php echo esc_attr($field).'_'.Filter::META_PREFIX.'_product_'.$categoryId; ?>"
                                            id="<?php echo esc_attr($field).'_'.Filter::META_PREFIX.'_product_'.$categoryId; ?>">
                                            <option value="">
                                                <?php echo _e('Default', 'heylight'); ?>
                                            </option>
                                            <?php foreach (Filter::PRODUCT_TYPES as $productType) : ?>
                                                <option data-pro="<?php echo ($productType['pro'] ? 'true' : 'false'); ?>" value="<?php echo $productType['id']; ?>"
                                                    <?php echo ($categoryData['filters'][Filter::META_PREFIX.'_product'] == $productType['id'] ? 'selected' : ''); ?>>
                                                    <?php echo $productType['name']; ?>
                                                </option>
                                            <?php endforeach; ?>
                                        </select>
                                    </label>
                                </td>

                                <td>
                                    <label style="display: block;" for="<?php echo esc_attr($field).'_'.Filter::META_PREFIX.'_code_'.$categoryId; ?>">
                                        <input type="text" name="<?php echo esc_attr($field).'_'.Filter::META_PREFIX.'_code_'.$categoryId; ?>" 
                                            id="<?php echo esc_attr($field).'_'.Filter::META_PREFIX.'_code_'.$categoryId; ?>" 
                                            value="<?php echo ($categoryData['filters'][Filter::META_PREFIX.'_code'] != null ? $categoryData['filters'][Filter::META_PREFIX.'_code'] : ''); ?>">
                                    </label>
                                </td>
                            <?php endif; ?>

                            <td>
                                <button type="button" class="delete-filter button" 
                                    onclick="document.getElementById('<?php echo esc_attr($field).'_'.Filter::META_PREFIX.'_disable_'.$categoryId; ?>').checked = false;
                                    document.getElementById('<?php echo esc_attr($field).'_'.Filter::META_PREFIX.'_apply_to_basket_'.$categoryId; ?>').checked = false;
                                    <?php if (Widget::CAN_BE_PRO) : ?>
                                        document.getElementById('<?php echo esc_attr($field).'_'.Filter::META_PREFIX.'_code_'.$categoryId; ?>').value = '';
                                        document.getElementById('<?php echo esc_attr($field).'_'.Filter::META_PREFIX.'_product_'.$categoryId; ?>').value = '';
                                    <?php endif; ?>
                                    <?php foreach ($options as $key => $name) : ?>
                                        document.getElementById('<?php echo esc_attr($field).'_'.Filter::META_PREFIX.'_terms_'.$categoryId.'_'.$key; ?>').checked = false;
                                    <?php endforeach; ?>
                                    document.getElementsByClassName('woocommerce-save-button')[0].click();">
                                    <?php _e('Elimina', 'heylight'); ?>
                                </button>
                            </td>
                        </tr>
                    <?php endforeach; ?>

                    <?php if (! empty($noFilterCategories)) : ?>
                    <tr class="category-filter">
                        <td class="category-name">
                            <select name="<?php echo esc_attr($field).'_'.Filter::META_PREFIX.'_category_id_new'; ?>">
                                <option value="" selected disabled>
                                    <?php _e('Scegli una categoria', 'heylight'); ?>
                                </option>
                                <?php foreach ($noFilterCategories as $key => $category) : ?>
                                    <option value="<?php echo $category->term_id; ?>">
                                        <?php echo $category->name; ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </td>

                        <td>
                            <label style="display: block;" for="<?php echo esc_attr($field).'_'.Filter::META_PREFIX.'_disable_new'; ?>">
                                <input type="checkbox" name="<?php echo esc_attr($field).'_'.Filter::META_PREFIX.'_disable_new'; ?>" 
                                    id="<?php echo esc_attr($field).'_'.Filter::META_PREFIX.'_disable_new'; ?>" />
                            </label>
                        </td>

                        <td>
                            <label style="display: block;" for="<?php echo esc_attr($field).'_'.Filter::META_PREFIX.'_apply_to_basket_new'; ?>">
                                <input type="checkbox" name="<?php echo esc_attr($field).'_'.Filter::META_PREFIX.'_apply_to_basket_new'; ?>" 
                                    id="<?php echo esc_attr($field).'_'.Filter::META_PREFIX.'_apply_to_basket_new'; ?>"/>
                            </label>
                        </td>

                        <td>
                            <div class="instalments-checkboxes">
                            <?php
                            $options = Gateway::getTermOptions();
                            foreach ($options as $key => $name) {
                                ?>
                                <div class="instalment-checkbox">
                                    <input type="checkbox" id="<?php echo esc_attr($field).'_'.Filter::META_PREFIX.'_terms_new_'.$key; ?>"
                                        value="<?php echo $key; ?>"
                                        data-pro="<?php echo (in_array($key, Widget::PRO_TERMS) ? 'true' : 'false'); ?>"
                                        name="<?php echo esc_attr($field).'_'.Filter::META_PREFIX.'_terms_new[]'; ?>"/>
                                    <label for="<?php echo esc_attr($field).'_'.Filter::META_PREFIX.'_terms_new_'.$key; ?>" title="<?php echo $name; ?>">
                                        <?php echo $key; ?>
                                    </label>
                                </div>
                                <?php
                            }
                            ?>
                            </div>
                        </td>

                        <?php if (Widget::CAN_BE_PRO) : ?>
                            <td>
                                <label style="display: block;" for="<?php echo esc_attr($field).'_'.Filter::META_PREFIX.'_product_new'; ?>">
                                    <select name="<?php echo esc_attr($field).'_'.Filter::META_PREFIX.'_product_new'; ?>"
                                        id="<?php echo esc_attr($field).'_'.Filter::META_PREFIX.'_product_new'; ?>">
                                        <option value="">
                                            <?php echo _e('Default', 'heylight'); ?>
                                        </option>
                                        <?php foreach (Filter::PRODUCT_TYPES as $productType) : ?>
                                            <option data-pro="<?php echo ($productType['pro'] ? 'true' : 'false'); ?>" value="<?php echo $productType['id']; ?>">
                                                <?php echo $productType['name']; ?>
                                            </option>
                                        <?php endforeach; ?>
                                    </select>
                                </label>
                            </td>

                            <td>
                                <label style="display: block;" for="<?php echo esc_attr($field).'_'.Filter::META_PREFIX.'_code_new'; ?>">
                                    <input type="text" name="<?php echo esc_attr($field).'_'.Filter::META_PREFIX.'_code_new'; ?>" 
                                        id="<?php echo esc_attr($field).'_'.Filter::META_PREFIX.'_code_new'; ?>" >
                                </label>
                            </td>
                        <?php endif; ?>

                    </tr>
                    <?php endif; ?>
                    </tbody>
                </table>
            </td>
        </tr>
        <?php
        return ob_get_clean();
    }

    /**
     * Save categories
     * @param array $postData 
     * @return void 
     */
    public function saveCategories($postData)
    {
        $prefix = 'woocommerce_'.'heylight'.'_categories_'.Filter::META_PREFIX.'_';

        /** Update categories */
        $categories = get_categories([
            'taxonomy' => 'product_cat',
            'orderby' => 'name',
            'show_count' => 0,
            'pad_counts' => 0,
            'hierarchical' => 1,
            'hide_empty' => 0
        ]);
        foreach ($categories as $category) {
            $filters = Filter::getFilterForCategory($category);
            if (! $filters) {
                continue;
            }
            $disable = (empty($postData[$prefix.'disable_'.$category->term_id]) || $postData[$prefix.'disable_'.$category->term_id] != 'on' ? false : true);
            $applyToBasket = (empty($postData[$prefix.'apply_to_basket_'.$category->term_id]) || $postData[$prefix.'apply_to_basket_'.$category->term_id] != 'on' ? false : true);
            $terms = (empty($postData[$prefix.'terms_'.$category->term_id]) ? [] : $postData[$prefix.'terms_'.$category->term_id]);
            $product = (empty($postData[$prefix.'product_'.$category->term_id]) ? '' : $postData[$prefix.'product_'.$category->term_id]);
            $code = (empty($postData[$prefix.'code_'.$category->term_id]) ? '' : $postData[$prefix.'code_'.$category->term_id]);

            if (count($terms) > Filter::MAX_INSTALMENTS) {
                WC_Admin_Settings::add_error(__('Maximum 6 choices for the number of instalments', 'heylight'));
                throw new Exception(__('Maximum 6 choices for the number of instalments', 'heylight'));
            }

            /** Remove category meta */
            if (! $disable && ! $applyToBasket && empty($terms) && empty($product) && empty($code)) {
                delete_term_meta($category->term_id, Filter::META_PREFIX.'_disable');
                delete_term_meta($category->term_id, Filter::META_PREFIX.'_apply_to_basket');
                delete_term_meta($category->term_id, Filter::META_PREFIX.'_terms');
                delete_term_meta($category->term_id, Filter::META_PREFIX.'_product');
                delete_term_meta($category->term_id, Filter::META_PREFIX.'_code');
                continue;
            }

            /** Update category meta */
            update_term_meta($category->term_id, Filter::META_PREFIX.'_disable', $disable);
            update_term_meta($category->term_id, Filter::META_PREFIX.'_apply_to_basket', $applyToBasket);
            update_term_meta($category->term_id, Filter::META_PREFIX.'_terms', json_encode($terms));
            update_term_meta($category->term_id, Filter::META_PREFIX.'_product', $product);
            update_term_meta($category->term_id, Filter::META_PREFIX.'_code', $code);
        }

        /** Check new category */
        $categoryId = (empty($postData[$prefix.'category_id_new']) ? false : $postData[$prefix.'category_id_new']);
        if (! $categoryId) {
            return;
        }
        $disable = (empty($postData[$prefix.'disable_new']) || $postData[$prefix.'disable_new'] != 'on' ? false : true);
        $applyToBasket = (empty($postData[$prefix.'apply_to_basket_new']) || $postData[$prefix.'apply_to_basket_new'] != 'on' ? false : true);
        $terms = (empty($postData[$prefix.'terms_new']) ? [] : $postData[$prefix.'terms_new']);
        $product = (empty($postData[$prefix.'product_new']) ? '' : $postData[$prefix.'product_new']);
        $code = (empty($postData[$prefix.'code_new']) ? '' : $postData[$prefix.'code_new']);

        if (count($terms) > Filter::MAX_INSTALMENTS) {
            WC_Admin_Settings::add_error(__('Maximum 6 choices for the number of instalments', 'heylight'));
            throw new Exception(__('Maximum 6 choices for the number of instalments', 'heylight'));
        }

        if (! $disable && ! $applyToBasket && empty($terms) && empty($product) && empty($code)) {
            return;
        }
        
        /** Save new category */
        update_term_meta($categoryId, Filter::META_PREFIX.'_disable', $disable);
        update_term_meta($categoryId, Filter::META_PREFIX.'_apply_to_basket', $applyToBasket);
        update_term_meta($categoryId, Filter::META_PREFIX.'_terms', json_encode($terms));
        update_term_meta($categoryId, Filter::META_PREFIX.'_product', $product);
        update_term_meta($categoryId, Filter::META_PREFIX.'_code', $code);
    }

    /**
     * Get term options
     * @return array 
     */
    public static function getTermOptions()
    {
        $termOptions = [];
        foreach (Widget::AVAILABLE_TERMS as $term) {
            $termOptions[$term] = $term . ' ' . __('rate', 'heylight');
        }
        foreach (Widget::PRO_TERMS as $proTerm) {
            if (isset($termOptions[$proTerm])) {
                continue;
            }
            $termOptions[$proTerm] = $proTerm . ' ' . __('rate', 'heylight');
        }
        asort($termOptions, SORT_NUMERIC);
        return $termOptions;
    }

    /**
     * Get product options
     * @return array 
     */
    public static function getProductOptions()
    {
        $options = [];
        foreach (Filter::PRODUCT_TYPES as $producType) {
            $options[$producType['id']] = $producType['name'];
        }
        return $options;
    }
}
